/* ---------------------------------------------------------------------
%   Copyright (C) 2007 Association for the COINS Compiler Infrastructure
%       (Read COPYING for detailed information.)
--------------------------------------------------------------------- */
package coins.driver;

import java.io.PrintStream;
import java.util.List;

/**
 *
 * Abstraction of a specification of compilation, i.e., specification of
 * options, arguments and source files.<br>
 *
 * Objects that implements this interface provide accesses to a compile
 * specification through this interface.  It may be generated by parsing
 * command line, may be by asking to an operator through GUI, or may be
 * generated from some database record automatically.<br>
 *
 * <i>option</i> may be one of:
 * <ul>
 *   <li> <tt>-E</tt><br>
 *        stop after preprocessing.
 *   <li> <tt>-S</tt><br>
 *        stop after assembly code generation.
 *   <li> <tt>-c</tt><br>
 *        do not link.
 *   <li> <tt>-help</tt><br>
 *        show help messages.
 *   <li> <tt>-o</tt> <i>file</i><br>
 *        specify an output file.
 *   <li> <tt>-b</tt><i>arch</i><br>
 *        specify a target architecture.
 *   <li> <tt>-pipe</tt><br>
 *        use pipes rather than temporary files to communicate between passes.
 *   <li> <tt>-W</tt><i>category</i><br>
 *        show warning messages of a category <i>category</i>
 *   <li> <tt>-Wno-</tt><i>category</i><br>
 *        don't show warning messages of a category <i>category</i>
 *   <li> <tt>-C</tt><br>
 *        preserve comments.
 *   <li> <tt>-D</tt><i>macro</i>[<tt>=</tt><i>definition</i>]<br>
 *        define a preprocessor macro.
 *   <li> <tt>-I</tt><i>path</i><br>
 *        specify an include path.
 *   <li> <tt>-P</tt><br>
 *        inhibit <tt>#line</tt> directives.
 *   <li> <tt>-U</tt><i>macro</i><br>
 *        undefine a preprocessor macro.
 *   <li> <tt>-O</tt><i>level</i><br>
 *        specify an optimization level to <i>level</i>
 *   <li> <tt>-g</tt><br>
 *        preserve symbols.
 *   <li> <tt>-p</tt><br>
 *        use performance monitor.
 *   <li> <tt>-L</tt><i>path</i><br>
 *        specify an archive search path.
 *   <li> <tt>-dynamic</tt><br>
 *        use dynamic linkage.
 *   <li> <tt>-static</tt><br>
 *        use static linkage.
 *   <li> <tt>-l</tt><i>archive</i><br>
 *        specify an archive to link.
 *   <li> <tt>-coins:</tt><i>options</i><br>
 *        coins options.
 * </ul>
 * Any other options are invalid and should be warned.
 *
 **/

public interface CompileSpecification {
  /* **************** DRIVER CONTROL OPTIONS **************** */

  /**
   *
   * Switch to stop after preprocessing and leave a preprocessed source code.
   * A driver control option.
   **/
  public static final String PREPROCESS_ONLY = "-E";

  /**
   * Switch to stop after assembly code generation and leave it.  A driver
   * control option.
   **/
  public static final String COMPILE_ONLY = "-S";

  /**
   * Switch not to link.  A driver control option.
   **/
  public static final String ASSEMBLE_ONLY = "-c";

  /**
   * Switch to show help.  A driver control option.
   **/
  public static final String HELP = "-help";

  /**
   * Option to specify an output file.  Takes an argument.  A driver control
   * option.
   **/
  public static final String OUTPUT_FILE = "-o";

  /**
   * Option to specify a target architecture.  May take an argument.  A driver
   * control option.
   **/
  public static final String TARGET_ARCHITECTURE = "-b";

  /**
   * Verbose option.
   **/
  public static final String VERBOSE = "-v";

  /**
   * Switch to use pipe rather than temporary files to communicate between
   * passes.  A driver control option.
   **/
  public static final String PIPE = "-pipe";


  /* **************** OVERALL OPTIONS **************** */

  /**
   * Option header to show warning messages of a category.  An overall option
   * which is passed to all passes.
   **/
  public static final String WARNING_CATEGORY = "-W";

  /* **************** PREPROCESSOR OPTIONS **************** */

  /**
   * Switch to preserve comments in output file.  A preprocessor option.
   **/
  public static final String PRESERVE_COMMENTS = "-C";

  /**
   * Option header to define a preprocessor macro.  A preprocessor option.
   **/
  public static final String DEFINE_MACRO = "-D";

  /**
   * Option header to specify an include path.  A preprocessor option.
   **/
  public static final String INCLUDE_PATH = "-I";

  /**
   * Option header not to output #line directives.  A preprocessor option.
   **/
  public static final String INHIBIT_NUMBER_LINE = "-P";

  /**
   * Option header to undefine a preprocessor macro.  A preprocessor option.
   **/
  public static final String UNDEFINE_MACRO = "-U";


  /* **************** COMPILER OPTIONS **************** */

  /**
   * Option header to set optimization level.  A compiler option.
   **/
  public static final String OPTIMIZE_LEVEL = "-O";

  /**
   * Option to preserve all symbols.  A compiler option.
   **/
  public static final String PRESERVE_SYMBOLS = "-g";

  /**
   * Option to use performance monitor.  A compiler option.
   **/
  public static final String PERFORMANCE_MONITOR = "-p";

  /* **************** LINKER OPTIONS **************** */

  /**
   * Option to specify an archive search path.  A linker option.
   **/
  public static final String LINK_PATH = "-L";

  /**
   * Option to use dynamic linkage.  A linker option.
   **/
  public static final String DYNAMIC_LINKAGE = "-dynamic";

  /**
   * Option to use static linkage.  A linker option.
   **/
  public static final String STATIC_LINKAGE = "-static";

  /**
   *
   * Option to speficy an archive to link.  Usually, specifying
   * "<tt>-l</tt><i>archive</i>" means to link archive file whose name is
   * <tt>lib</tt><i>archive</i><tt>.a</tt>.  A linker option.
   **/
  public static final String ARCHIVE_TO_LINK = "-l";

  /* **************** others **************** */

  /**
   *
   * Option header of COINS options.  Even if more than one COINS options are
   * specified, they are interpreted as a single COINS option whose argument
   * is arguments of the options concatenated with commas.
   **/
  public static final String COINS = "-coins";

  /* **************** methods **************** */

  /**
   * Tests if an option is set or not in this command line.
   *
   * @param option the option string, including leading `-'.
   * @return <tt>true</tt> if the option is set, <tt>false</tt> otherwise.
   */
  public boolean isSet(String option);

  /**
   * Returns an argument of an option in this command line.  When the option
   * is not set, a <tt>null</tt> is returned.  The argument is a
   * <tt>String</tt> for <tt>-o</tt>, a <tt>List</tt> of <tt>String</tt>s for
   * <tt>-D</tt>, <tt>-I</tt>, <tt>-L</tt>, <tt>-O</tt>, <tt>-U</tt>,
   * <tt>-W</tt> and <tt>-l</tt>.
   *
   * @param option the option string, including leading `-'.
   * @return the argument of <tt>option</tt>.
   **/
  public Object getArg(String option);

  /**
   *
   * Tests if the argument string is one of preprocessor options.  The
   * argument string should start with `-' to be judged as a preprocessor
   * option.
   *
   * @param arg the argument string
   * @return true if `arg' is one of the preprocessor options; false otherwise.
   **/
  public boolean isPreprocessorOption(String arg);

  /**
   *
   * Returns a <tt>List</tt> of <tt>String</tt>s containing all preprocessor
   * options specified in this command line.  One copy of the list is created
   * per a call.
   *
   * @return preprocessor options
   **/
  public List getPreprocessorOptions();

  /**
   *
   * Tests if the argument string is one of compiler options.  The argument
   * string should start with `-' to be judged as a compiler option.
   *
   * @param arg the argument string
   * @return true if `arg' is one of the compiler options; false otherwise.
   **/
  public boolean isCompilerOption(String arg);

  /**
   *
   * Returns a <tt>List</tt> of <tt>String</tt>s containing all compiler
   * options specified in this command line.  One copy of the list is created
   * per a call.
   *
   * @return compiler options
   **/
  public List getCompilerOptions();

  /**
   *
   * Tests if the argument string is one of assembler options.  The argument
   * string should start with `-' to be judged as a assembler option.
   *
   * @param arg the argument string
   * @return true if `arg' is one of the assembler options; false otherwise.
   **/
  public boolean isAssemblerOption(String arg);

  /**
   *
   * Returns a <tt>List</tt> of <tt>String</tt>s containing all assembler
   * options specified in this command line.  One copy of the list is created
   * per a call.
   *
   * @return assembler options.
   **/
  public List getAssemblerOptions();

  /**
   *
   * Tests if the argument string is one of linker options.  The argument
   * string should start with `-' to be judged as a linker option.
   *
   * @param arg the argument string
   * @return true if `arg' is one of the linker options; false otherwise.
   **/
  public boolean isLinkerOption(String arg);

  /**
   *
   * Returns a <tt>List</tt> of <tt>String</tt>s containing all linker options
   * specified in this command line.  One copy of the list is created per a
   * call.
   *
   * @return linker options
   **/
  public List getLinkerOptions();

  /**
   * Returns an CoinsOptions object which represents -coins options included
   * in this command line.
   *
   * @return a CoinsOptions object.
   **/
  public CoinsOptions getCoinsOptions();

  /**
   * Returns a <tt>List</tt> of source file names listed in this command line.
   * One copy of the list is created per a call.
   *
   * @return the <tt>List</tt> of source files names.
   **/
  public List getSourceFiles();

  /**
   * Registers a file as an object file of a specified source file.  Without
   * calling this method, getLinkerOptions cannot return a correct command
   * line arguments.
   *
   * @param sourceFile the source file name
   * @param objectFile the object file name
   * @throws Error there is no source file sourceFile
   **/
  public void setObjectFile(String sourceFile, String objectFile);

  /**
   * Returns a Trace object which filters trace messages in a manner
   * specified in this command line.
   *
   * @return a Trace object.
   */
  public Trace getTrace();

  /**
   * Returns a Warning object which filters warning messages in a manner
   * specified in this command line.
   *
   * @return a Warning object.
   */
  public Warning getWarning();

  /**
   * Shows help messages.
   *
   * @param out A PrintStream to which the help messages are written.
   * @param driver A driver object which has the method `main'.
   **/
  public void showHelp(PrintStream out, CompilerImplementation driver);
}
