package coins.backend;

import java.lang.*;
import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.sym.*;
import coins.backend.lir.*;
import coins.backend.util.*;

/**
 * Represent module, the whole input of the compiler.
 */
public class Module {
  /** Module name */
  public final String name;

  /** Global symbol table */
  public final SymTab globalSymtab = new SymTab();

  /** List of functions */
  public final BiList functionList = new BiList();

  /** Create a module */
  public Module(Object sexp, PrintWriter output) throws SyntaxError {
    if (!(sexp instanceof ImList))
      throw new SyntaxError("Expected (");
    // output.println("Module:doCodegen() sexp = " + (ImList)sexp);
    ImList ptr = (ImList)sexp;
    if (ptr.elem() != Keyword.MODULE)
      throw new SyntaxError("Expected MODULE");
    if ((ptr = ptr.next()).atEnd())
      throw new SyntaxError("Expected module name");

    name = (String)ptr.elem();

    // Parse top-level S-expressions
    while (!(ptr = ptr.next()).atEnd()) {
      ImList top = (ImList)ptr.elem();
      String kw = (String)top.elem();
      if (kw == Keyword.SYMTAB)
        doSymtbl(top);
      else if (kw == Keyword.DATA)
        doData(top);
      else if (kw == Keyword.FUNCTION) {
        Function f = new Function(this, top, output);
        functionList.add(f);
        if (Debug.dumpFunction) {
          output.println("After CFG created:");
          f.printIt(output);
        }
      }
      else
        throw new SyntaxError("Unexpected " + kw);
    }
  }

  private void doSymtbl(ImList aTop) throws SyntaxError {

    // Parse symbol table
    while (!(aTop = aTop.next()).atEnd()) {
      ImList sym = (ImList)aTop.elem();
      String name = (String)sym.elem();
      int storage = Storage.decode((String)sym.elem2nd());
      int type = Type.decode((String)sym.elem3rd());
      int boundary = Integer.parseInt((String)sym.elem4th());
      if (storage == Storage.STATIC) {
        globalSymtab.addSymbol(name, storage, type, boundary,
                               (String)sym.elem5th(),
                               (String)sym.elem6th());
      } else {
        globalSymtab.addSymbol(name, storage, type, boundary,
                               Integer.parseInt((String)sym.elem5th()));
      }
    }
  }

  private void doData(ImList aTop) throws SyntaxError {}


  /** Apply some analysis on each function */
  public void apply(LocalAnalyzer analyzer) {
    for (BiLink p = functionList.first(); !p.atEnd(); p = p.next())
      ((Function)p.elem()).apply(analyzer);
  }

  /** Require some analysis for each function */
  public void require(LocalAnalyzer analyzer) {
    for (BiLink p = functionList.first(); !p.atEnd(); p = p.next())
      ((Function)p.elem()).require(analyzer);
  }

  /** Do some transform(or optimization) on each function */
  public void apply(LocalTransform transformer) {
    for (BiLink p = functionList.first(); !p.atEnd(); p = p.next())
      ((Function)p.elem()).apply(transformer);
  }

  /** Print current module status */
  public void printIt(PrintWriter out) { printIt(out, null); }

  /** Print current module status with analyses */
  public void printIt(PrintWriter out, LocalAnalyzer[] anals) {
    out.println("Module \"" + name + "\":");
    out.print("Global ");
    globalSymtab.printIt(out);
    for (BiLink p = functionList.first(); !p.atEnd(); p = p.next()) {
      ((Function)p.elem()).printIt(out, anals);
    }
  }
}
