package coins.backend;

import java.lang.*;
import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.util.*;
import coins.backend.sym.*;
import coins.backend.lir.*;
import coins.backend.cfg.*;

/** Represent a function. */
public class Function implements Keywords, LirDef {
  /** Module this function belongs to */
  public final Module module;

  /** Symbol table entry of this function */
  public final SymStatic symbol;

  /** Local symbol table */
  public final SymTab localSymtab = new SymTab();

  /** Label table */
  private Map labelTable = new HashMap();

  /** LIR Node generator */
  public final LirFactory newLir = new LirFactory();

  /** Control flow graph */
  public final FlowGraph flowGraph;

  private Dominators dominators;

  /** Parse S-expression function description and convert to internal form */
  public Function(Module mod, ImList ptr, PrintWriter output)
    throws SyntaxError {
    module = mod;

    // Parse name of the function
    ptr = ptr.next();
    symbol = (SymStatic)module.globalSymtab.get((String)ptr.elem());
    symbol.setBody(this);

    // Parse symbol table
    ptr = ptr.next();
    ImList symp = (ImList)ptr.elem();
    if (symp.elem() != KW_SYMTAB)
      throw new SyntaxError("SYMTAB expected");
    for (symp = symp.next(); !symp.atEnd(); symp = symp.next()) {
      ImList sym = (ImList)symp.elem();
      localSymtab.addSymbol((String)sym.elem(),
                            LStorage.decode((String)sym.elem2nd()),
                            LType.decode((String)sym.elem3rd()),
                            Integer.parseInt((String)sym.elem4th()),
                            Integer.parseInt((String)sym.elem5th()));
    }

    // Parse body part
    BiList instructions = new BiList();
    while (!(ptr = ptr.next()).atEnd()) {
      ImList stmt = (ImList)ptr.elem();
      LirNode lir = decodeLir(stmt);
      instructions.add(lir);
    }

    if (Debug.dumpAfterRead) {
      // Dump function
      output.println("Function Just after read:");
      output.println("(FUNCTION \"" + symbol.name + "\"");
      output.print("  (SYMTAB");
      Iterator it = localSymtab.iterator();
      while (it.hasNext()) {
        output.println();
        output.print("    " + (Symbol)it.next());
      }
      output.println(" )");
      for (BiLink p = instructions.first(); !p.atEnd(); p = p.next())
        output.println("    " + (LirNode)p.elem());
      output.println(")");
    }

    // Make control flow graph
    flowGraph = new FlowGraph(this, instructions);

    // Dominators
    dominators = new Dominators(flowGraph);
  }


  private LirNode decodeLir(ImList stmt) throws SyntaxError {
    int code = OpTable.toCode((String)stmt.elem());

    switch (code) {
        // leaf nodes
    case OP_INTCONST:
      return newLir.iconst(LType.decode((String)stmt.elem2nd()),
                           Integer.parseInt((String)stmt.elem3rd()));
    case OP_FLOATCONST:
      return newLir.fconst(LType.decode((String)stmt.elem2nd()),
                           Double.parseDouble((String)stmt.elem3rd()));
    case OP_STATIC:
    case OP_FRAME:
    case OP_REG:
    case OP_SUBREG:
      {
        int type = LType.decode((String)stmt.elem2nd());
        String name = (String)stmt.elem3rd();
        Symbol sym = localSymtab.get(name);
        if (sym == null) {
            sym = module.globalSymtab.get(name);
            if (sym == null)
                throw new Error("Undefined symbol: " + name);
        }
        return newLir.symRef(code, type, sym);
      }

      // Labels are interned
    case OP_DEFLABEL:
      {
        String label = (String)stmt.elem2nd();
        LirNode it = (LirNode)labelTable.get(label);
        if (it == null) {
          it = newLir.defLabel(label);
          labelTable.put(label, it);
        }
        return it;
      }

    case OP_LABEL:
      {
        String label = (String)stmt.elem3rd();
        LirNode it = (LirNode)labelTable.get(label);
        if (it == null) {
          it = newLir.defLabel(label);
          labelTable.put(label, it);
        }
        return newLir.operator(OP_LABEL, TY_ADDRESS, it);
      }

    // unary operators with type
    case OP_NEG:
    case OP_CONVSX:
    case OP_CONVZX:
    case OP_CONVIT:
    case OP_CONVFX:
    case OP_CONVFT:
    case OP_CONVFI:
    case OP_CONVSF:
    case OP_CONVUF:
    case OP_MEM:
      return newLir.operator(code, LType.decode((String)stmt.elem2nd()),
                             decodeLir((ImList)stmt.elem3rd()));

    // unary operators w/o type
    case OP_JUMP:
    case OP_USE:
    case OP_CLOBBER:
      return newLir.operator(code, TY_UNKNOWN, decodeLir((ImList)stmt.elem2nd()));

    // binary operators
    case OP_ADD:
    case OP_SUB:
    case OP_MUL:
    case OP_DIVS:
    case OP_DIVU:
    case OP_MODS:
    case OP_MODU:
    case OP_BAND:
    case OP_BOR:
    case OP_BXOR:
    case OP_BNOT:
    case OP_LSHS:
    case OP_LSHU:
    case OP_RSHS:
    case OP_RSHU:
    case OP_TSTEQ:
    case OP_TSTNE:
    case OP_TSTLTS:
    case OP_TSTLES:
    case OP_TSTGTS:
    case OP_TSTGES:
    case OP_TSTLTU:
    case OP_TSTLEU:
    case OP_TSTGTU:
    case OP_TSTGEU:
    case OP_SET:
      return newLir.operator(code, LType.decode((String)stmt.elem2nd()),
                             decodeLir((ImList)stmt.elem3rd()),
                             decodeLir((ImList)stmt.elem4th()));

    case OP_JUMPC: // ternary operator
      {
        LirNode[] src = new LirNode[3];
        src[0] = decodeLir((ImList)stmt.elem2nd());
        src[1] = decodeLir((ImList)stmt.elem3rd());
        src[2] = decodeLir((ImList)stmt.elem4th());
        return newLir.operator(code, TY_UNKNOWN, src);
      }
        
    case OP_CALL: // currently with type
      {
        int type = LType.decode((String)stmt.elem2nd());
        LirNode callee = decodeLir((ImList)stmt.elem3rd());
        int n = ((ImList)stmt.elem4th()).length(); // parameter list length
        LirNode[] param = new LirNode[n];
        int i = 0;
        for (ImList p = (ImList)stmt.elem4th(); !p.atEnd(); p = p.next())
          param[i++] = decodeLir((ImList)p.elem());
        return newLir.operator(code, type, callee,
                               newLir.operator(OP_LIST, TY_UNKNOWN, param));
      }

    case OP_JUMPN:
      {
        LirNode value = decodeLir((ImList)stmt.elem2nd());
        ImList cases = (ImList)stmt.elem3rd();
        int n = cases.length();
        LirNode[] labels = new LirNode[n];
        int i = 0;
        for (ImList p = cases; !p.atEnd(); p = p.next()) {
          ImList c = (ImList)p.elem();
          labels[i++] = newLir.operator(OP_LIST, TY_UNKNOWN,
                                        decodeLir((ImList)c.elem()),
                                        decodeLir((ImList)c.elem2nd()));
        }
        return newLir.operator(code, TY_UNKNOWN, value,
                               newLir.operator(OP_LIST, TY_UNKNOWN, labels),
                               decodeLir((ImList)stmt.elem4th()));
      }
        
    case OP_PROLOGUE:
    case OP_EPILOGUE:
      {
        int n = stmt.next().length();
        LirNode[] opr = new LirNode[n];
        ImList frame = (ImList)stmt.elem2nd();
        opr[0] = newLir.operator(OP_LIST, TY_UNKNOWN,
                                 newLir.iconst(TY_ADDRESS,
                                               Integer.parseInt((String)frame.elem())),
                                 newLir.iconst(TY_ADDRESS,
                                               Integer.parseInt((String)frame.elem2nd())));
        int i = 1;
        for (ImList p = stmt.next().next(); !p.atEnd(); p = p.next())
          opr[i++] = decodeLir((ImList)p.elem());
        return newLir.operator(code, TY_UNKNOWN, opr);
      }

    case OP_PARALLEL:
      {
        int n = stmt.next().length();
        LirNode[] src = new LirNode[n];
        int i = 0;
        for (ImList p = stmt.next(); !p.atEnd(); p = p.next())
          src[i++] = decodeLir((ImList)p.elem());
        return newLir.operator(code, TY_UNKNOWN, src);
      }

    default:
      throw new Error("Unknown opCode");
    }

  }

  /** Dump internal data structure of the Function object. */
  public void printIt(PrintWriter out) {
    out.println();
    out.println("Function \"" + symbol.name + "\":");
    out.print(" Local ");
    localSymtab.printIt(out);
    out.println();
    out.println("Control Flow Graph:");
    flowGraph.printIt(out);

    if (dominators != null)
      dominators.printIt(out);
  }

}
