package coins.backend.cfg;

import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.lir.*;
import coins.backend.util.*;

/** Represent Control Flow Graph, a directed graph whose
 *   nodes are basic blocks. */
public class FlowGraph implements LirDef {
  /** Function owning this CFG. */
  public final Function function;

  /** List of basic blocks. */
  public final BiList basicBlkList = new BiList();

  /** Identifier of the next allocated block */
  private int blkCounter = 1;

  /** Split LIR instruction list into basic blocks
   *   and build up CFG. */
  public FlowGraph(Function f, BiList anInstrList) {
    function = f;

    // Partition instructions into basic blocks.
    boolean prevJump = false;
    boolean prevLabel = false;
    BiLink left;
    for (BiLink ptr = anInstrList.first(); !ptr.atEnd(); ptr = left) {
      left = ptr.next();
      LirNode node = (LirNode)ptr.elem();
      node.setRoot(true);
      if (prevJump || node.opCode == OP_DEFLABEL && !prevLabel) {
        BiList fragment = anInstrList;
        anInstrList = anInstrList.split(ptr);
        if (!prevJump) {
          // add explicit JUMP instruction goes to next block
          fragment.add(function.newLir.operator(OP_JUMP, TY_UNKNOWN,
                        function.newLir.operator(OP_LABEL, TY_ADDRESS, node)));
        }
        basicBlkList.add(newBasicBlk(fragment));
      }
      switch (node.opCode) {
      case OP_DEFLABEL:
        prevLabel = true;
        prevJump = false;
        break;
      case OP_JUMP:
      case OP_JUMPC:
      case OP_JUMPN:
        prevJump = true;
        prevLabel = false;
        break;
      default:
        prevLabel = prevJump = false;
        break;
      }
    }
    if (!anInstrList.isEmpty()) {
      // last fragment
      basicBlkList.add(newBasicBlk(anInstrList));
    }

    // Add edges
    for (BiLink bp = basicBlkList.first(); !bp.atEnd(); bp = bp.next()) {
      BasicBlk blk = (BasicBlk)bp.elem();
      blk.maintEdges();
    }

    dfstOrder(null);
  }


  /** Create new basic block with instruction list <code>instr</code>. */
  public BasicBlk newBasicBlk(BiList instr) {
    return new BasicBlk(this, blkCounter++, instr);
  }


  /** Return maximum block numer + 1. */
  public int maxBlkIdent() { return blkCounter; }
  

  /** Depth First Ordering */
  public void dfstOrder(DfstHook h) {
    for (BiLink p = basicBlkList.first(); !p.atEnd(); p = p.next()) {
      BasicBlk x = (BasicBlk)p.elem();
      x.rpostNum = x.preNum = 0;
      x.parent = null;
    }

    int[] cpre = new int[1];
    int[] crpost = new int[1];

    entryBlk().depthFirstSearch(h, null, cpre, crpost);

    int bias = -crpost[0];
    for (BiLink p = basicBlkList.first(); !p.atEnd(); p = p.next()) {
      BasicBlk x = (BasicBlk)p.elem();
      if (x.preNum != 0)
        x.rpostNum += bias;
    }
  }


  /** Return the entry basic block. */
  public BasicBlk entryBlk() { return (BasicBlk)basicBlkList.first().elem(); }

  /** Return the list of basic blocks. */
  // public BiList basicBlkList() { return basicBlkList; }

  /** Return iterator for accessing basic blocks. */
  public Iterator basicBlkIterator() { return basicBlkList.iterator(); }

  /** Print CFG */
  public void printIt(PrintWriter output) {
    for (BiLink bp = basicBlkList.first(); !bp.atEnd(); bp = bp.next()) {
      BasicBlk blk = (BasicBlk)bp.elem();
      blk.printIt(output);
    }
  }

}
