package coins.backend;

import java.lang.*;
import java.io.*;
import coins.backend.*;
import coins.backend.lir.*;

/**
 * LIR type encoder/decoder without instance.
 * Type is represented by an int, not an instance of the class.
 */
public class Type {

  public static final int UNKNOWN = 0;
  public static final int INT = 1;
  public static final int FLOAT = 2;
  public static final int AGGREGATE = 3;
  public static final int ADDRESS = INT + 32; // ##Should consult TMD##


  private static int TAGWIDTH = 3;
  private static int TAGMASK = ((1 << TAGWIDTH) - 1);

  // Inhibit invokation of constructor.
  private Type() {}

  public static String toString(int type) {
    int bits = type & ~TAGMASK;
    switch (type & TAGMASK) {
    case UNKNOWN:
      return Keyword.UNKNOWN;
    case INT:
      return "I" + bits;
    case FLOAT:
      return "F" + bits;
    case AGGREGATE:
      return "A" + bits;
    default:
      throw new IllegalArgumentException("bad type");
    }
  }

  public static int tag(int type) {
    return type & TAGMASK;
  }

  public static int bits(int type) {
    return type & ~TAGMASK;
  }

  public static boolean isAggregate(int type) {
    return (type & TAGMASK) == UNKNOWN;
  }

  public static int decode(String word) throws SyntaxError {
    if (word == Keyword.UNKNOWN)
      return UNKNOWN;
    else if (word.charAt(0) == 'I') {
      int bits = Integer.parseInt(word.substring(1));
      return INT + ((bits + TAGMASK) & ~TAGMASK);
    }
    else if (word.charAt(0) == 'F') {
      int bits = Integer.parseInt(word.substring(1));
      return FLOAT + ((bits + TAGMASK) & ~TAGMASK);
    }
    else if (word.charAt(0) == 'A') {
      int bits = Integer.parseInt(word.substring(1));
      return AGGREGATE + ((bits + TAGMASK) & ~TAGMASK);
    }
    else
      throw new SyntaxError("Type expected but " + word);
  }
}
