package coins.backend.lir;

import coins.backend.*;
import coins.backend.util.*;
import coins.backend.sym.*;
import coins.backend.cfg.*;

/**
 * New LIR node's basic structure
 *  
 */
public abstract class LirNode {
  /** Key to bind related information */
  public final int id;

  /** Instruction Code */
  public final int opCode;

  /** Type of this node's result */
  public final int type;

  /** Optional arguments list (op opr1 opr2 &hogehoge ...) */
  public final ImList opt;

  // int flags;
  // static final int F_ROOT = 1;
  
  /** Create LIR node */
  LirNode(int id, int opCode, int type, ImList opt) {
    this.id = id;
    this.opCode = opCode;
    this.type = type;
    this.opt = (opt == null) ? ImList.Empty : opt;
  }

  /** Return a copy of this node */
  public abstract LirNode makeCopy(LirFactory fac);


  /** Return number of operands */
  public int nSrcs() { return 0; }

  /** Return nth operand; Subclass responsibility */
  public LirNode src(int n) { throw new IllegalArgumentException(); }

  /** Set nth operand; Subclass responsibility */
  public void setSrc(int n, LirNode src) {
    throw new IllegalArgumentException();
  }

  /** Return jump target labels. Targets are packed in an array of Label. */
  public Label[] getTargets() { return null; }

  /** Replace target label y for x. */
  public void replaceLabel(Label x, Label y, LirFactory fac) {
    throw new IllegalArgumentException();
  }

  /** Return true if this node is the root of the tree. */
  // public boolean isRoot() { return (flags & F_ROOT) != 0; }

  /** Set true/false on root flag */
  // public void setRoot(boolean yes) {
  // if (yes)
  // flags |= F_ROOT;
  // else
  // flags &= ~F_ROOT;
  // }


  /** Traverse the tree of LirNode in preorder. **/
  public void walkPreorder(LirVisitor v) {
    accept(v);
    int n = nSrcs();
    for (int i = 0; i < n; i++)
      src(i).walkPreorder(v);
  }

  /** Traverse the tree of LirNode in postorder. **/
  public void walkPostorder(LirVisitor v) {
    int n = nSrcs();
    for (int i = 0; i < n; i++)
      src(i).walkPostorder(v);
    accept(v);
  }




  /** Return true if node is a physical register.
      (to be overriden in subclasses) **/
  public boolean isPhysicalRegister() { return false; }


  /** Is register operand? **/
  private boolean isRegisterOperand() {
    return (opCode == Op.REG
            || opCode == Op.SUBREG && src(0).opCode == Op.REG);
  }

  /** Pick up def'd/used register variables in the LirNode tree. **/
  public void pickupDefUseReg(DefUseHandler handler) {
    switch (opCode) {
    case Op.PARALLEL:
      int n = nSrcs();
      for (int i = 0; i < n; i++) {
        switch (src(i).opCode) {
        case Op.SET:
        case Op.CLOBBER:
          if (src(i).src(0).isRegisterOperand())
            handler.defined(src(i).src(0));
          break;
        }
      }
      for (int i = 0; i < n; i++) {
        switch (src(i).opCode) {
        case Op.SET:
          if (!src(i).src(0).isRegisterOperand())
            src(i).src(0).pickupDefUseReg(handler);
          src(i).src(1).pickupDefUseReg(handler);
          break;
        case Op.CLOBBER:
          break;
        default:
          {
            int m = src(i).nSrcs();
            for (int j = 0; j < m; j++)
              src(i).src(j).pickupDefUseReg(handler);
          }
          break;
        }
      }
      break;

    case Op.SET:
      if (src(0).isRegisterOperand())
        handler.defined(src(0));
      else
        src(0).pickupDefUseReg(handler);
      src(1).pickupDefUseReg(handler);
      break;

    case Op.CLOBBER:
      if (src(0).isRegisterOperand())
        handler.defined(src(0));
      break;

    case Op.CALL:
      // return values
      n = src(2).nSrcs();
      for (int i = 0; i < n; i++) {
        if (src(2).src(i).isRegisterOperand())
          handler.defined(src(2).src(i));
      }
      // other operands (callee and parameters)
      src(1).pickupDefUseReg(handler);
      src(2).pickupDefUseReg(handler);
      break;

    case Op.PROLOGUE:
      n = nSrcs();
      for (int i = 1; i < n; i++) {
        if (src(i).isRegisterOperand())
          handler.defined(src(i));
      }
      break;

    default:
      if (isRegisterOperand())
        handler.used(this);
      else {
        int m = nSrcs();
        for (int j = 0; j < m; j++)
          src(j).pickupDefUseReg(handler);
      }
      break;
    }
  }





  /** Visualize */
  public String toString() {
    StringBuffer buf = new StringBuffer();
    if (Debug.showId)
      buf.append("$" + id + ": ");
    buf.append("(");
    if (opCode != Op.LIST) {
      buf.append(Op.toName(opCode));
      if (type != Type.UNKNOWN) {
        buf.append(Debug.TypePrefix);
        buf.append(Type.toString(type));
      }
      buf.append(" ");
    }
    int n = nSrcs();
    int i = 0;
    if (opCode == Op.PROLOGUE || opCode == Op.EPILOGUE) {
      // handle first operand specially
      buf.append("(" + ((LirIconst)src(0).src(0)).value);
      buf.append(" " + ((LirIconst)src(0).src(1)).value);
      buf.append(")");
      i = 1;
    }
    for (; i < n; i++) {
      if (i != 0)
        buf.append(" ");
      buf.append(src(i).toString());
    }
    buf.append(")");
    return buf.toString();
  }

  /** Return true if this object equals to x */
  public boolean equals(Object x) {
    return (x instanceof LirNode
            && ((LirNode)x).opCode == opCode && ((LirNode)x).type == type
            && opt.equals(((LirNode)x).opt));
  }

  /** Accept visitor v */
  public abstract void accept(LirVisitor v);

}
