package coins.backend.opt;

import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.util.*;
import coins.backend.sym.*;
import coins.backend.lir.*;
import coins.backend.cfg.*;
import coins.backend.ana.*;


/** Transform into SSA (Static Single Assignment) form. */
public class Ssa implements LocalTransform {
  private PrintWriter output;

  private Function function;
  private Dominators dom;

  public Ssa(PrintWriter out) {
    output = out;
  }

  /** Rewrite to SSA form. */
  public void doIt(Function f) {
    function = f;
    FlowGraph flowGraph = f.flowGraph;
    DominanceFrontiers df = (DominanceFrontiers)f.require(DominanceFrontiers.analyzer);
    dom = (Dominators)f.require(Dominators.analyzer);

    dom.printIt(output);

    int symBound = f.localSymtab.idBound();
    int insBound = f.newLir.idBound();

    // Collect assignments to variables.
    BiList[] assigns = new BiList[symBound]; // List of BasicBlk
    Symbol[] symbols = new Symbol[symBound];

    for (BiLink p = flowGraph.basicBlkList.first(); !p.atEnd(); p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();
      for (BiLink q = blk.instrList().first(); !q.atEnd(); q = q.next()) {
        LirNode ins = (LirNode)q.elem();
        if (ins.opCode == Op.SET && ins.src(0).opCode == Op.REG) {
          Symbol sym = ((LirSymRef)ins.src(0)).symbol;
          if (symbols[sym.id] == null) {
            symbols[sym.id] = sym;
            assigns[sym.id] = new BiList();
          }
          assigns[sym.id].addNew(blk);
        }
      }
    }

    // Insert PHI instructions.
    BasicBlk[] worklist = new BasicBlk[insBound];
    int[] inWork = new int[insBound];
    int[] hasAlready = new int[insBound];
    int nwork = 0;
    int iterCount = 0;
    for (int i = 1; i < symBound; i++) {
      if (assigns[i] != null) {
        iterCount++;
        for (BiLink p = assigns[i].first(); !p.atEnd(); p = p.next()) {
          BasicBlk blk = (BasicBlk)p.elem();
          inWork[blk.id] = iterCount;
          worklist[nwork++] = blk;
        }
        while (nwork > 0) {
          BasicBlk blk = worklist[--nwork];
          for (BiLink q = df.frontiers[blk.id].first();
               !q.atEnd(); q = q.next()) {
            BasicBlk front = (BasicBlk)q.elem();
            if (hasAlready[front.id] < iterCount) {
              // insert PHI at front
              int nPreds = front.predList().length() + 1;
              LirNode[] operand = new LirNode[nPreds];
              operand[0] = f.newLir.symRef(Op.REG, symbols[i].type, symbols[i]);
              for (int j = 1; j < nPreds; j++)
                operand[j] = operand[0];
              front.instrList().addFirst(f.newLir.operator(Op.PHI, symbols[i].type, operand));
              hasAlready[front.id] = iterCount;
              if (inWork[front.id] < iterCount) {
                inWork[front.id] = iterCount;
                worklist[nwork++] = front;
              }
            }
          }
        }
      }
    }

    output.println("After PHI insertion:");
    flowGraph.printIt(output);

    // Rename variable references.
    LirNode undefinedValue = function.newLir
      .operator(Op.UNDEFINED, Type.UNKNOWN, new LirNode[0]);
    int[] nextVar = new int[symBound];
    LirNode[] curVar = new LirNode[symBound];
    for (int i = 0; i < symBound; i++)
      curVar[i] = undefinedValue;
    renameInBlock(flowGraph.entryBlk(), nextVar, curVar);
  }

  // Replace variables in block blk.
  private void renameInBlock(BasicBlk blk, int[] nextVar, LirNode[] curVar) {
    class Stack {
      Stack prev;
      Symbol orig;
      LirNode top;
      Stack(Stack prev, Symbol orig, LirNode top) {
        this.prev = prev;
        this.orig = orig;
        this.top = top;
      }
    }

    Stack stackPtr = null;

    output.println("** Entering block #" + blk.id);

    // Replace in this block
    for (BiLink q = blk.instrList().first(); !q.atEnd(); q = q.next()) {
      LirNode ins = (LirNode)q.elem();
      // replace all (REG v) occurence in right hand side
      replaceRhs(ins, curVar);

      // replace left hand side
      switch (ins.opCode) {
      case Op.PHI:
      case Op.SET:
        {
          LirNode lhs = ins.src(0);
          if (lhs.opCode == Op.REG) {
            Symbol origVar = ((LirSymRef)lhs).symbol;
            stackPtr = new Stack(stackPtr, origVar, curVar[origVar.id]);
            ins.setSrc(0, createNewVar(origVar, nextVar, curVar));
          }
        }
        break;

      case Op.PROLOGUE:
        int n = ins.nSrcs();
        for (int i = 1; i < n; i++) {
          LirNode lhs = ins.src(i);
          if (lhs.opCode == Op.REG) {
            Symbol origVar = ((LirSymRef)lhs).symbol;
            stackPtr = new Stack(stackPtr, origVar, curVar[origVar.id]);
            ins.setSrc(i, createNewVar(origVar, nextVar, curVar));
          }
        }
        break;
      }
    }

    // Replace variables propagating to successor's PHI instructions.
    for (BiLink s = blk.succList().first(); !s.atEnd(); s = s.next()) {
      BasicBlk succ = (BasicBlk)s.elem();
      int n = succ.predList().whereIs(blk);
      
      for (BiLink q = succ.instrList().first(); !q.atEnd(); q = q.next()) {
        LirNode ins = (LirNode)q.elem();
        if (ins.opCode == Op.PHI) {
          if (ins.src(1 + n).opCode != Op.REG)
            throw new IllegalArgumentException();
          Symbol origVar = ((LirSymRef)ins.src(1 + n)).symbol;
          output.println("** replace PHI: " + origVar.name + " to "
                         + curVar[origVar.id]);
          ins.setSrc(1 + n, function.newLir
                     .operator(Op.LIST, Type.UNKNOWN,
                               curVar[origVar.id],
                               function.newLir.labelRef(Op.LABEL, Type.ADDRESS,
                                                        blk.label())));
        }
      }
    }

    // Replace in immediate dominatee blocks.
    for (BiLink p = dom.kids[blk.id].first(); !p.atEnd(); p = p.next()) {
      BasicBlk kid = (BasicBlk)p.elem();
      renameInBlock(kid, nextVar, curVar);
    }

    // Restore curVar as first state upon entry.
    while (stackPtr != null) {
      curVar[stackPtr.orig.id] = stackPtr.top;
      stackPtr = stackPtr.prev;
    }
  }

  // Replace variables in right hand side
  private LirNode replaceRhs(LirNode ins, LirNode[] curVar) {
    switch (ins.opCode) {
    case Op.REG:
      return curVar[((LirSymRef)ins).symbol.id];
    case Op.PHI:
    case Op.PROLOGUE:
      break;
    case Op.SET:
      replaceRhs(ins.src(0), curVar);
      ins.setSrc(1, replaceRhs(ins.src(1), curVar));
      break;
    default:
      int n = ins.nSrcs();
      for (int i = 0; i < n; i++)
        ins.setSrc(i, replaceRhs(ins.src(i), curVar));
      break;
    }
    return ins;
  }

  // Introduce new SSA variable.
  private LirNode createNewVar(Symbol origVar, int[] nextVar, LirNode[] curVar) {
    Symbol newVar = function.localSymtab
      .addSymbol(origVar.name + "." +  nextVar[origVar.id],
                 Storage.REG, origVar.type, 0, 0);
    nextVar[origVar.id]++;
    newVar.setShadow(origVar);
    output.println("** replace LHS: " + origVar.name + " to " + newVar.name);
    curVar[origVar.id] = function.newLir.symRef(Op.REG, newVar.type, newVar);
    return curVar[origVar.id];
  }
}

