package coins.backend;

import java.lang.*;
import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.util.*;
import coins.backend.tmd.*;
import coins.backend.sym.*;

/** Target Description Module **/
public class TargetMachine {

  private TMD tmd;

  /** Collection of global variables **/
  private BackEnd root;

  /** Table for (def name value) **/
  private Map definitions = new HashMap();

  /** Table for register sets **/
  private Map regsetTable = new HashMap();

  /** Physical register variable's base **/
  private int regbase;

  /** Physical register table **/
  private Symbol[] physicalRegisters;

  /** Initialize Target Description information **/
  public TargetMachine(SymTab table, String tmdfile, BackEnd root)
    throws SyntaxError, IOException {

    tmd = new TMD(tmdfile);
    this.root = root;

    // Read machine dependent parameters
    PushbackReader rdr = new PushbackReader(new StringReader(tmd.params()));

    Object sexp;
    while ((sexp = ImList.readSexp(rdr)) != null) {
      if (!(sexp instanceof ImList))
        throw new SyntaxError("Non-list object in TMD parameters");

      ImList p = (ImList)sexp;
      if ((String)p.elem() != "def")
        throw new SyntaxError("def expected but " + (String)p.elem());

      String defname = (String)p.elem2nd();
      Object value = p.elem3rd();
      definitions.put(defname, value);

      if (defname == "*real-register-symtab*") {
        // Parse physical register table
        p = (ImList)value;
        if (p.elem() != "SYMTAB")
          throw new SyntaxError("SYMTAB expected bug " + p.elem());

        p = p.next();
        physicalRegisters = new Symbol[p.length()];
        regbase = table.idBound();
        int i = 0;
        for (; !p.atEnd(); p = p.next())
          physicalRegisters[i++] = table.addSymbol((ImList)p.elem());

        table.printIt(root.debOut);
      }
      else if (defname == "predefined-parameter-foo") {
      }
      else if (defname == "predefined-parameter-bar") {
      }
      else if (defname.matches("\\*regset-.*\\*")) {
        // Register set definition
        long set = 0;
        for (p = (ImList)value; !p.atEnd(); p = p.next()) {
          // Parse (REG type "%rname")
          ImList node = (ImList)p.elem();
          if (node.elem() == "SUBREG")
            node = (ImList)node.elem2nd();
          if (node.elem() != "REG")
            throw new SyntaxError("REG exptected but " + node.elem());
          int type = Type.decode((String)node.elem2nd());
          String name = ((QuotedString)node.elem3rd()).body;
          Symbol reg = table.get(name);
          if (reg == null)
            throw new SyntaxError("Undeclared REG: " + name);
          set |= 1 << regNumber(reg);
        }
        regsetTable.put(defname, new Long(set));
      }
      else
        throw new SyntaxError("Unknown def-parameter in TMD: " + defname);
    }
  }


  /** Return number of physical registers **/
  public int nRegisters() { return physicalRegisters.length; }


  /** Convert symbol entry to physical register number. **/
  public int regNumber(Symbol phyReg) {
    if (phyReg.name.charAt(0) != '%')
      throw new IllegalArgumentException("non-physical register");
    return phyReg.id - regbase;
  }


  /** Convert physical register number to symbol entry. **/
  public Symbol numberReg(int number) {
    return physicalRegisters[number];
  }


  /** Return register set **/
  public long getRegset(String name) {
    Long v = (Long)regsetTable.get(name);
    if (v == null)
      throw new IllegalArgumentException("Undefined regset: " + name);
    return v.longValue();
  }


  /** Return the value of a def'ed name. **/
  public Object getDef(String name) {
    return definitions.get(name);
  }

  /** Convert function func to machine dependent form. **/
  public Function instSel(Function func) throws SyntaxError, IOException {
    ByteArrayOutputStream bytes = new ByteArrayOutputStream();
    PrintWriter out = new PrintWriter(bytes);
    func.printStandardForm(out);
    out.close();

    root.debOut.println("Before instsel: ");
    root.debOut.print(bytes.toString());
    root.debOut.flush();

    String rewrittenCode = tmd.restra(bytes.toString());
    String machineDepStr = tmd.instsel(rewrittenCode);

    // root.debOut.println("After instsel: ");
    // root.debOut.print(machineDepStr);
    // root.debOut.flush();

    // Reload matched code.
    Function machineDep;
    Object sexp = ImList.readSexp(new PushbackReader
                                    (new StringReader(machineDepStr)));
    if (!(sexp instanceof ImList))
      throw new CantHappenException("readSexp returns null or atom object");

    return new Function(func.module, (ImList)sexp, root);
  }


  /** Assembler output **/
  public void asmout(Module mod, PrintWriter outputfile) {
  }

}
