package coins.backend.ana;

import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.cfg.*;
import coins.backend.lir.*;
import coins.backend.sym.*;
import coins.backend.util.*;


/** Create interference graph **/
public class InterferenceGraph implements LocalAnalysis {

  /** Factory class of Dominators. */
  private static class Analyzer implements LocalAnalyzer {
    public LocalAnalysis doIt(Function func) {
      return new InterferenceGraph(func);
    }
  }

  /** Factory singleton. */
  public static final Analyzer analyzer = new Analyzer();


  private Function function;
  private int timeStamp;

  private BitSet adjMatrix;

  private int nSyms;
  private Symbol[] symVec;
  private BiList[] interfereLists;

  public InterferenceGraph(Function func) {
    function = func;
    timeStamp = func.flowGraph.timeStamp();

    nSyms = func.localSymtab.idBound();
    symVec = func.localSymtab.sortedSymbols();
    adjMatrix = new BitSet(nSyms * (nSyms - 1) / 2);
    interfereLists = new BiList[nSyms];
    for (int i = 1; i < nSyms; i++)
      interfereLists[i] = new BiList();

    LiveVariableAnalysis liveInfo
      = (LiveVariableAnalysis)func.require(LiveVariableAnalysis.analyzer);

    for (BiLink p = func.flowGraph.basicBlkList.first(); !p.atEnd();
         p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();

      final BitSet live = liveInfo.liveOutBitSet(blk);

      DefUseHandler handler = new DefUseHandler() {
          public void defined(LirNode node) {
            if (node.opCode == Op.SUBREG)
              node = node.src(0);
            int defvar = ((LirSymRef)node).symbol.id;
            for (int i = 0; (i = live.nextSetBit(i)) >= 0; i++)
              setInterfere(defvar, i);
          }

          public void used(LirNode node) {}
        };

      for (BiLink q = blk.instrList().last(); !q.atEnd(); q = q.prev()) {
        LirNode stmt = (LirNode)q.elem();

        if (stmt.opCode == Op.PHI) {
          handler.defined(stmt.src(0));
        } else {
          LiveVariableAnalysis.pickupDefUseReg(stmt, handler);
          liveInfo.maintainLiveOverStmt(live, stmt);
        }
      }
    }
  }

  private void setInterfere(int x, int y) {
    if (x != y) {
      if (x < y) { int w = x; x = y; y = w; }
      if (!adjMatrix.get(x * (x - 1) / 2 + y)) {
        adjMatrix.set(x * (x - 1) / 2 + y);
        interfereLists[x].add(symVec[y]);
        interfereLists[y].add(symVec[x]);
      }
    }
  }


  public boolean interfere(Symbol x, Symbol y) {
    return adjMatrix.get(x.id * (x.id - 1) / 2 + y.id);
  }


  public BiList interfereList(Symbol x) {
    return interfereLists[x.id];
  }


  /** Return true if this analysis is up to date. */
  public boolean isUpToDate() {
    return (timeStamp == function.flowGraph.timeStamp());
  }


  /** Debug print entries required by interface. **/

  public void printBeforeFunction(PrintWriter output) {}
  public void printBeforeBlock(BasicBlk blk, PrintWriter output) {}
  public void printAfterBlock(BasicBlk blk, PrintWriter output) {}
  public void printBeforeStmt(LirNode stmt, PrintWriter output) {}
  public void printAfterStmt(LirNode stmt, PrintWriter output) {}

  public void printAfterFunction(PrintWriter output) {
    output.println();
    output.println("Interference Graph:");
    for (int i = 1; i < nSyms; i++) {
      output.print(symVec[i].name + ":");
      for (BiLink p = interfereLists[i].first(); !p.atEnd(); p = p.next()) {
        Symbol x = (Symbol)p.elem();
        output.print(" " + x.name);
      }
      output.println();
    }
  }

  public void printIt(PrintWriter output) {
    printAfterFunction(output);
  }

}
