package coins.backend.cfg;

import java.io.*;
import coins.backend.lir.*;
import coins.backend.util.*;

/** Represent basic block, a sequence of LIR instructions
 * without intervening JUMPs. */
public class BasicBlk implements LirDef {
  /** FlowGraph owning this block. */
  public final FlowGraph flowGraph;

  /** Identifier of this block. */
  public final int ident;

  /** Instructions list */
  private BiList instrList;

  /** Successors of this block. */
  private BiList succList = new BiList();

  /** Predecessors of this basic block. */
  private BiList predList = new BiList();

  /** DFST Reverse postorder number */
  int rpostNum;

  /** DFST Preorder number */
  int preNum;

  /** Parent node in the DFST. */
  BasicBlk parent;

  /** Create basic block with the fragment of instruction list.
   *  Called only by FlowGraph. */
  BasicBlk(FlowGraph g, int id, BiList anInstrList) {
    flowGraph = g;
    ident = id;
    instrList = anInstrList;

    // Make DEFLABELs belong to this block
    for (BiLink insp = instrList.first(); !insp.atEnd(); insp = insp.next()) {
      LirNode node = (LirNode)insp.elem();
      if (node.opCode == OP_DEFLABEL)
        ((LirDefLabel)node).setBasicBlk(this);
    }
  }

  /** Return instruction list */
  public BiList instrList() { return instrList; }

  /** Return instruction list */
  void setInstrList(BiList list) { instrList = list; }

  /** Return list of successors */
  public BiList succList() { return succList; }

  /** Return list of predecessors */
  public BiList predList() { return predList; }

  /** Add an edge from this block to toBlk */
  void addEdge(BasicBlk toBlk) {
    succList.addNew(toBlk);
    toBlk.predList.addNew(this);
  }

  /** Remove the edge from this block to toBlk */
  void removeEdge(BasicBlk toBlk) {
    succList.remove(toBlk);
    toBlk.predList.remove(toBlk);
  }

  /** Remove all edges from this block */
  void clearEdges() {
    for (BiLink p = succList.first(); !p.atEnd(); ) {
      BasicBlk target = (BasicBlk)p.elem();
      p = p.next();
      target.predList.remove(this);
    }
    succList.clear();
  }

  /** Maintain edges from this block.
   * Automatically checks JUMP(/C/N) instructions  */
  public void maintEdges() {
    clearEdges();

    // check last instruction
    if (!instrList.isEmpty()) {
      LirNode[] targets = ((LirNode)instrList.last().elem()).getTargets();
      if (targets != null) {
        for (int i = 0; i < targets.length; i++)
          addEdge(((LirDefLabel)targets[i]).basicBlk());
      }
    }
  }

  /** Depth First Search */
  void depthFirstSearch(DfstHook h, BasicBlk from, int[] cpre, int[] crpost) {
    if (h != null)
      h.preOrder(this, from);
    preNum = ++cpre[0];
    parent = from;
    for (BiLink p = succList.first(); !p.atEnd(); p = p.next()) {
      BasicBlk y = (BasicBlk)p.elem();
      if (y.preNum == 0)
        y.depthFirstSearch(h, this, cpre, crpost);
    }
    if (h != null)
      h.postOrder(this);
    rpostNum = crpost[0]--;
  }

  /** Print this block */
  public void printIt(PrintWriter output) {
    output.println();
    output.print("  #" + ident + " Basic Block: DFO=" + rpostNum + ",");
    if (parent != null)
      output.print(" parent=#" + parent.ident + ",");
    output.print(" <-(");
    boolean listContinue = false;
    for (BiLink blkp = predList.first(); !blkp.atEnd(); blkp = blkp.next()) {
      BasicBlk blk = (BasicBlk)blkp.elem();
      output.print((listContinue ? ",#" : "#") + blk.ident);
      listContinue = true;
    }
    output.print(") ->(");
    listContinue = false;
    for (BiLink blkp = succList.first(); !blkp.atEnd(); blkp = blkp.next()) {
      BasicBlk blk = (BasicBlk)blkp.elem();
      output.print((listContinue ? ",#" : "#") + blk.ident);
      listContinue = true;
    }
    output.println(")");
    for (BiLink insp = instrList.first(); !insp.atEnd(); insp = insp.next()) {
      LirNode node = (LirNode)insp.elem();
      output.print("    ");
      output.println(node.toString());
    }
  }

}
