package coins.backend.lir;

import coins.backend.*;
import coins.backend.cfg.*;

/**
 * New LIR node's basic structure
 *  
 */
public abstract class LirNode implements LirDef {
  /** Key index to bind related information */
  public final int ident;

  /** Instruction Code */
  public final int opCode;

  /** Type of this node's result */
  public final int type;

  /** Miscellaneous attributes */
  int flags;
  static final int F_ROOT = 1;
  
  /** Create LIR node */
  LirNode(int id, int op, int t) {
    ident = id;
    opCode = op;
    type = t;
  }

  /** Return number of operands */
  public int nSrcs() { return 0; }

  /** Return nth operand; Subclass responsibility */
  public LirNode src(int n) { throw new IllegalArgumentException(); }

  /** Set nth operand; Subclass responsibility */
  public void setSrc(int n, LirNode src) {
    throw new IllegalArgumentException();
  }

  /** Return jump target labels (array of DEFLABEL nodes) */
  public LirNode[] getTargets() { return null; }

  /** Return true if this node is the root of the tree. */
  public boolean isRoot() { return (flags & F_ROOT) != 0; }

  /** Set true/false on root flag */
  public void setRoot(boolean yes) {
    if (yes)
      flags |= F_ROOT;
    else
      flags &= ~F_ROOT;
  }

  /** Visualize */
  public String toString() {
    StringBuffer buf = new StringBuffer();
    //    if (ident != 0)
    //      buf.append("$" + ident + ": ");
    buf.append("(");
    if (opCode != OP_LIST) {
      buf.append(OpTable.toName(opCode));
      if (type != TY_UNKNOWN || opCode == OP_CALL) {
        buf.append(TypePrefix);
        buf.append(LType.toString(type));
      }
      buf.append(" ");
    }
    if (opCode == OP_LABEL) {
      // omit DEFLABEL
      buf.append("\"" + ((LirDefLabel)src(0)).name + "\"");
    } else {
      int n = nSrcs();
      for (int i = 0; i < n; i++) {
        if (i != 0)
          buf.append(" ");
        buf.append(src(i).toString());
      }
    }
    buf.append(")");
    return buf.toString();
  }

  /** Accept visitor v */
  public abstract void accept(LirVisitor v);

}
