package coins.backend.util;

// Lisp-like object handling for Java.
//
// Cons Cell Object

import java.lang.*;
import java.io.*;
import coins.backend.*;

/**
 * Immutable uni-directional list.
 * Use BiList if you want mutable list.
 */
public class ImList extends Object {
  private Object elem;
  private ImList next;

  public static final ImList Empty = new ImList(null, null);

  /** Create a link */
  public ImList(Object content, ImList tail) {
    elem = content;
    next = tail;
  }

  /** Return the list header */
  public ImList next() { return next; }

  /** Return first element of the list. */
  public Object elem() { return elem;  }

  /** Return second element of the list. */
  public Object elem2nd() { return next.elem; }

  /** Return third element of the list. */
  public Object elem3rd() { return next.next.elem; }

  /** Return fourth element of the list. */
  public Object elem4th() { return next.next.next.elem; }

  /** Return fifth element of the list. */
  public Object elem5th() { return next.next.next.next.elem; }

  /** Return sixth element of the list. */
  public Object elem6th() { return next.next.next.next.next.elem; }

  /** Return true if this list is at end (empty). */
  public boolean atEnd() { return this == Empty; }

  /** Return number of elements of the list. */
  public int length() {
    int n = 0;
    for (ImList p = this; p != Empty; p = p.next)
      n++;
    return n;
  }

  /** Reverse the list destructively. Think three times before use. */
  public ImList destructiveReverse() {
    ImList p = this, q = Empty;
    while (p != Empty) {
      ImList w = p.next;
      p.next = q;
      q = p;
      p = w;
    }
    return q;
  }


  /** Read S-expression from Reader prd and build them up in ImList form.
   *  @return the list or String object. null indicates EOF. */
  public static Object readSexp(PushbackReader prd)
    throws IOException, SyntaxError {
    int c;

    c = skipSpaces(prd);
    if (c < 0)
      return null;

    if (c == '(') {
      ImList ptr = Empty;
      for (;;) {
        c = skipSpaces(prd);
        if (c < 0)
          throw new SyntaxError("Unexpected EOF");
        if (c == ')')
          break;
        prd.unread(c);
        ptr = new ImList(readSexp(prd), ptr);
      }
      return ptr.destructiveReverse();
    }

    else if (c == ')')
      throw new SyntaxError("Invalid )");

    else if (c == '.')
      throw new SyntaxError("Invalid .");

    else if (c == '"') {
      // String
      StringBuffer buf = new StringBuffer();
      boolean escape = false;

      for (;;) {
        if ((c = prd.read()) < 0)
          break;
        if (escape) {
          escape = false;
        } else {
          if (c == '"')
            break;
          if (c == '\\') {
            escape = true;
            continue;
          }
        }
        buf.append((char)c);
      }
      return buf.toString().intern();
    }
    else {
      // symbol or number atom
      StringBuffer buf = new StringBuffer();
      for (;;) {
        buf.append((char)c);
        if ((c = prd.read()) < 0)
          break;
        if (Character.isWhitespace((char)c) || c == ')' || c == '(') {
          prd.unread(c);
          break;
        }
      }
      return buf.toString().intern();
    }
  }

  private static int skipSpaces(PushbackReader prd)
    throws IOException {
    int c;

    for (;;) {
      // Skip whitespaces
      while ((c = prd.read()) >= 0 && Character.isWhitespace((char)c))
        ;
      if (c != ';')
        return c;

      // Skip comments until end of line
      while ((c = prd.read()) >= 0 && c != '\n')
        ;
    }
  }


  /** Visualize */
  public String toString() {
    StringBuffer buf = new StringBuffer();
    buf.append("(");
    boolean first = true;
    for (ImList p = this; p != Empty; p = p.next) {
      if (!first) buf.append(" ");
      buf.append(p.elem.toString());
      first = false;
    }
    buf.append(")");
    return buf.toString();
  }
}
