package coins.backend.ana;

import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.cfg.*;
import coins.backend.util.*;

/** Find dominators of the control flow graph. */
public class Dominators implements LocalAnalysis {

  /** Factory class of Dominators. */
  private static class Analyzer implements LocalAnalyzer {
    public LocalAnalysis doIt(Function func) {
      return new Dominators(func.flowGraph);
    }
  }

  /** Factory singleton. */
  public static final Analyzer analyzer = new Analyzer();

  /** BasicBlk-id-indexed array, whose elements are the parents
   *    of the block in the dominator tree. */
  public final BasicBlk [] idom;

  /** BasicBlk-id-indexed array, whose elements are the lists of children
   *   nodes in the dominator tree. */
  public final BiList [] kids;

  /** Flow Graph */
  private FlowGraph flowGraph;

  /** Copy of CFG timestamp to be analyzed. */
  private int timeStamp;

  /** Create dominator tree for Graph g. */
  public Dominators(FlowGraph g) {
    flowGraph = g;
    timeStamp = flowGraph.timeStamp();

    int maxDfn = g.maxDfn();
    int idBound = g.idBound();

    // allocate widom
    int[] widom = new int[maxDfn + 1];
    BasicBlk [] blkdfn = new BasicBlk[maxDfn + 1];
    for (BiLink p = g.basicBlkList.first(); !p.atEnd(); p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();
      if (blk.dfn() != 0) {
        blkdfn[blk.dfn()] = blk;
        if (blk.parent() != null)
          widom[blk.dfn()] = blk.parent().dfn();
      }
    }

    // Compute Dominators
    boolean change;
    do {
      change = false;
      for (int i = 2; i <= maxDfn; i++) {
        int x = widom[i];
        for (BiLink p = blkdfn[i].predList().first(); !p.atEnd();
             p = p.next()) {
          int y = ((BasicBlk)p.elem()).dfn();
          while (x != y) {
            if (x > y)
              x = widom[x];
            else /* if (y > x) */
              y = widom[y];
          }
        }
        if (x != widom[i] && x != 0) {
          widom[i] = x;
          change = true;
        }
      }
    } while (change);

    // Now widom represents dominator tree.

    // Copy widom to idom.
    idom = new BasicBlk[idBound];
    for (int i = 1; i <= maxDfn; i++)
      idom[blkdfn[i].id] = blkdfn[widom[i]];

    // Set up children
    kids = new BiList[idBound];
    for (BiLink p = g.basicBlkList.first(); !p.atEnd(); p = p.next())
      kids[((BasicBlk)p.elem()).id] = new BiList();
    for (int i = 1; i <= maxDfn; i++) {
      if (widom[i] != 0)
        kids[blkdfn[widom[i]].id].add(blkdfn[i]);
    }
  }

  /** Return true if this analysis is up to date. */
  public boolean isUpToDate() {
    return (timeStamp == flowGraph.timeStamp());
  }

  /** Return immediate dominator of block blk. */
  public BasicBlk immDominator(BasicBlk blk) {
    return idom[blk.id];
  }

  /** Return children (immediate dominatees) list iterator of block blk. */
  public Iterator children(BasicBlk blk) {
    return kids[blk.id].iterator();
  }


  /** Test if the block x dominates y. */
  public boolean dominates(BasicBlk x, BasicBlk y) {
    // This is naive implementation: had better use O(1) algorithm.
    while (y.dfn() < x.dfn())
      y = idom[y.id];
    return y == x;
  }


  /** Dump dominator tree on stream out */
  public void printIt(PrintWriter out) {
    out.println();
    out.println("Dominator Tree:");
    dumpIt(flowGraph.entryBlk(), "", "", out);
  }

  private void dumpIt(BasicBlk blk, String pref1, String pref2, PrintWriter out) {
    out.print(pref1);
    if (pref2.length() > 0)
      out.print(" +-");
    out.print("#" + blk.id);
    if (!kids[blk.id].isEmpty())
      out.print("'s children: ");
    boolean first = true;
    for (BiLink p = kids[blk.id].first(); !p.atEnd(); p = p.next()) {
      out.print(first ? "#" : ",#");
      out.print(((BasicBlk)p.elem()).id);
      first = false;
    }
    out.println();
    String pref = pref1 + pref2;
    for (BiLink p = kids[blk.id].first(); !p.atEnd(); p = p.next())
      dumpIt((BasicBlk)p.elem(), pref, p.next().atEnd() ? "   " : " | ", out);
  }
}
