package coins.backend.ana;

import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.cfg.*;
import coins.backend.util.*;

/** Detect loop structures in the flow graph. <br>
 *   Algorithm used is taken from:<br>
 *    R. Endre Tarjan, "Testing Flow Graph Reducibility", 
 *    Journal of Computer and System Sciences 9, 355-365 (1974). */
public class LoopAnalysis implements LocalAnalysis {

  /** Factory class of Dominators. */
  private static class Analyzer implements LocalAnalyzer {
    public LocalAnalysis doIt(Function func) {
      return new LoopAnalysis(func.flowGraph);
    }
  }

  public static final Analyzer analyzer = new Analyzer();

  /** BasicBlk-id-indexed array, whose elements are the flags
   *   indicating whether the block is a loop header or not. */
  public boolean[] isLoop;

  /** BasicBlk-id-indexed array, whose elements are the block's entry
   *   block if it is a part of the loop, null otherwise. */
  public BasicBlk[] loopHeader;

  /** BasicBlk-id-indexed array, whose elements are the flags
   * indicating whether the loop is reducible or not. */
  public boolean[] multiEntry;

  /** BasicBlk-id-indexed array of loop nest depth. */
  public int[] nestLevel;

  /** Flow graph analyzed. */
  private FlowGraph flowGraph;

  private int timeStamp;

  /** Return the name of set which x belongs to.
   *  (Tarjan's UNION-FIND Path Compression Algorithm) */
  private static int find(int[] parent, int[] name, int x)
  {
    int y, w;

    /* y = root of x */
    for (y = x; parent[y] != 0; y = parent[y])
      ;
    /* compress */
    for (; (w = parent[x]) != 0; x = w)
      parent[x] = y;
    return name[y];
  }

  /** Disjoint Set Union: z = x U y */
  private static void union(int[] parent, int[] name, int[] root, int[] cnt,
                            int x, int y, int z)
  {
    int w, rx, ry;

    if (cnt[root[x]] < cnt[root[y]]) {
      w = x; x = y; y = w;
    }
    rx = root[x];
    ry = root[y];
    parent[ry] = rx;
    cnt[rx] += cnt[ry];
    name[rx] = z;
    root[z] = rx;
  }


  /** Find loops in the graph g */
  public LoopAnalysis(FlowGraph g) {
    flowGraph = g;
    timeStamp = g.timeStamp();

    int n = g.maxDfn();
    int[] added = new int[n + 1];
    int[] parent = new int[n + 1];
    int[] name = new int[n + 1];
    int[] root = new int[n + 1];
    int[] cnt = new int[n + 1];
    int[] body = new int[n + 1];
    int[] worklist = new int[n + 1];

    loopHeader = new BasicBlk[g.idBound()];
    isLoop = new boolean[g.idBound()];
    multiEntry = new boolean[g.idBound()];
    nestLevel = new int[g.idBound()];

    BasicBlk[] v = g.blkVectorByPre();

    for (int w = 1; w <= n; w++) {
	parent[w] = 0;
	name[w] = root[w] = w;
	cnt[w] = 1;
	added[w] = 0;
    }

    /* For each w in g (reverse preorder) */
    for (int w = n; w >= 1; w--) {
      /* collect nodes x such that there exists retreating edge x->w */
      int nb = 0, nw = 0;
      for (BiLink s = v[w].predList().first(); !s.atEnd(); s = s.next()) {
        int x = ((BasicBlk)s.elem()).dfnPre();
        if (x == 0) continue;
        if (v[x].dfn() >= v[w].dfn()) {   /* is a retreating edge? */
          int xx = find(parent, name, x);
          if (added[xx] != w) {
            worklist[nw++] = body[nb++] = xx;
            added[xx] = w;
          }
	}
      }
      if (nw > 0)
        isLoop[v[w].id] = true;

      while (nw > 0) {
        int x = worklist[--nw];
        for (BiLink s = v[x].predList().first(); !s.atEnd(); s = s.next()) {
          int y = ((BasicBlk)s.elem()).dfnPre();
          if (y == 0) continue;
          /* tree/advancing/cross edge? */
          if (v[y].dfn() < v[x].dfn()) {
            int yy = find(parent, name, y);
            /* is yy a descendant of w? */
            if (w <= yy && v[w].dfn() <= v[yy].dfn()) {
              if (added[y] != w && yy != w) {
                worklist[nw++] = body[nb++] = yy;
                added[yy] = w;
              }
            } else {
              /* yy is out of loop; so x is another entry of the loop. */
              /* nonreducible - multi entry loop */
              multiEntry[v[w].id] = true;
            }
          }
        }
      }
      while (nb > 0) {
        int x = body[--nb];
        loopHeader[v[x].id] = v[w];
        union(parent, name, root, cnt, x, w, w);
      }
    }


    // Count nest level
    for (int w = 1; w <= n; w++) {
      BasicBlk blk = v[w];
      if (loopHeader[blk.id] != null)
        nestLevel[blk.id] = nestLevel[loopHeader[blk.id].id] + 1;
      if (isLoop[blk.id])
        nestLevel[blk.id]++;
    }

  }


  
  /** Return true if this analysis is up to date. */
  public boolean isUpToDate() {
    return (timeStamp == flowGraph.timeStamp());
  }


  /** Print loop structure. */
  public void printIt(PrintWriter out) {
    int n = flowGraph.maxDfn();
    BasicBlk[] v = flowGraph.blkVectorByPre();

    out.println("Loop Structure:");
    for (int i = 1; i <= n; i++) {
      if (isLoop[v[i].id]) {
        for (int k = 0; k < nestLevel[v[i].id]; k++)
          out.print("  ");
        out.print("Loop starts at #" + v[i].id + ": ");
        boolean first = true;
        for (int j = i + 1; j <= n; j++) {
          if (loopHeader[v[j].id] != null && loopHeader[v[j].id].id == v[i].id) {
            if (!first) out.print(",");
            out.print("#" + v[j].id);
            first = false;
          }
        }
        out.println();
      }
    }
  }
}
