package coins.backend;

import java.lang.*;
import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.util.*;
import coins.backend.sym.*;
import coins.backend.lir.*;
import coins.backend.cfg.*;
import coins.backend.ana.*;

/** Represent a function. */
public class Function {
  /** Module this function belongs to */
  public final Module module;

  /** Symbol table entry of this function */
  public final SymStatic symbol;

  /** Local symbol table */
  public final SymTab localSymtab = new SymTab();

  /** Label table */
  public final Map labelTable = new HashMap();

  /** LIR Node generator */
  public final LirFactory newLir = new LirFactory();

  /** Control flow graph */
  public final FlowGraph flowGraph;

  /** Result of analyses, instances of LocalAnalysis */
  private Map analyses = new HashMap();

  private PrintWriter debOut;

  /** Parse S-expression function description and convert to internal form */
  public Function(Module mod, ImList ptr, PrintWriter output)
    throws SyntaxError {
    module = mod;
    debOut = output;

    // Parse name of the function
    ptr = ptr.next();
    symbol = (SymStatic)module.globalSymtab.get((String)ptr.elem());
    symbol.setBody(this);

    // Parse symbol table
    ptr = ptr.next();
    ImList symp = (ImList)ptr.elem();
    if (symp.elem() != Keyword.SYMTAB)
      throw new SyntaxError("SYMTAB expected");
    for (symp = symp.next(); !symp.atEnd(); symp = symp.next()) {
      ImList sym = (ImList)symp.elem();
      localSymtab.addSymbol((String)sym.elem(),
                            Storage.decode((String)sym.elem2nd()),
                            Type.decode((String)sym.elem3rd()),
                            Integer.parseInt((String)sym.elem4th()),
                            Integer.parseInt((String)sym.elem5th()));
    }

    // Parse body part
    BiList instructions = new BiList();
    while (!(ptr = ptr.next()).atEnd()) {
      ImList stmt = (ImList)ptr.elem();
      LirNode lir = decodeLir(stmt);
      instructions.add(lir);
    }

    if (Debug.dumpAfterRead) {
      // Dump function
      output.println("Function Just after read:");
      output.println("(FUNCTION \"" + symbol.name + "\"");
      output.print("  (SYMTAB");
      Iterator it = localSymtab.iterator();
      while (it.hasNext()) {
        output.println();
        output.print("    " + (Symbol)it.next());
      }
      output.println(" )");
      for (BiLink p = instructions.first(); !p.atEnd(); p = p.next())
        output.println("    " + (LirNode)p.elem());
      output.println(")");
    }

    // Make control flow graph
    flowGraph = new FlowGraph(this, instructions);
  }


  private LirNode decodeLir(ImList stmt) throws SyntaxError {
    int code = Op.toCode((String)stmt.elem());

    switch (code) {
        // leaf nodes
    case Op.INTCONST:
      return newLir.iconst(Type.decode((String)stmt.elem2nd()),
                           Integer.parseInt((String)stmt.elem3rd()));
    case Op.FLOATCONST:
      return newLir.fconst(Type.decode((String)stmt.elem2nd()),
                           Double.parseDouble((String)stmt.elem3rd()));
    case Op.STATIC:
    case Op.FRAME:
    case Op.REG:
      {
        int type = Type.decode((String)stmt.elem2nd());
        String name = (String)stmt.elem3rd();
        Symbol sym = localSymtab.get(name);
        if (sym == null) {
            sym = module.globalSymtab.get(name);
            if (sym == null)
                throw new Error("Undefined symbol: " + name);
        }
        return newLir.symRef(code, type, sym);
      }

    case Op.SUBREG:
      {
        int type = Type.decode((String)stmt.elem2nd());
        LirNode src = decodeLir((ImList)stmt.elem3rd());
        int pos = Integer.parseInt((String)stmt.elem4th());
        return newLir.subReg(Op.SUBREG, type, src, pos);
      }

      // Label reference/definition
    case Op.LABEL:
      {
        String name = (String)stmt.elem3rd();
        Label label = (Label)labelTable.get(name);
        if (label == null) {
          label = new Label(name);
          labelTable.put(name, label);
        }
        return newLir.labelRef(code, Type.ADDRESS, label);
      }

    case Op.DEFLABEL:
      {
        String name = (String)stmt.elem2nd();
        Label label = (Label)labelTable.get(name);
        if (label == null) {
          label = new Label(name);
          labelTable.put(name, label);
        }
        return newLir.labelRef(code, Type.UNKNOWN, label);
      }

    // unary operators with type
    case Op.NEG:
    case Op.BNOT:
    case Op.CONVSX:
    case Op.CONVZX:
    case Op.CONVIT:
    case Op.CONVFX:
    case Op.CONVFT:
    case Op.CONVFI:
    case Op.CONVSF:
    case Op.CONVUF:
    case Op.MEM:
      return newLir.operator(code, Type.decode((String)stmt.elem2nd()),
                             decodeLir((ImList)stmt.elem3rd()));

    // unary operators w/o type
    case Op.JUMP:
    case Op.USE:
    case Op.CLOBBER:
      return newLir.operator(code, Type.UNKNOWN, decodeLir((ImList)stmt.elem2nd()));

    // binary operators
    case Op.ADD:
    case Op.SUB:
    case Op.MUL:
    case Op.DIVS:
    case Op.DIVU:
    case Op.MODS:
    case Op.MODU:
    case Op.BAND:
    case Op.BOR:
    case Op.BXOR:
    case Op.LSHS:
    case Op.LSHU:
    case Op.RSHS:
    case Op.RSHU:
    case Op.TSTEQ:
    case Op.TSTNE:
    case Op.TSTLTS:
    case Op.TSTLES:
    case Op.TSTGTS:
    case Op.TSTGES:
    case Op.TSTLTU:
    case Op.TSTLEU:
    case Op.TSTGTU:
    case Op.TSTGEU:
    case Op.SET:
      return newLir.operator(code, Type.decode((String)stmt.elem2nd()),
                             decodeLir((ImList)stmt.elem3rd()),
                             decodeLir((ImList)stmt.elem4th()));

    case Op.JUMPC: // ternary operator
      {
        LirNode[] src = new LirNode[3];
        src[0] = decodeLir((ImList)stmt.elem2nd());
        src[1] = decodeLir((ImList)stmt.elem3rd());
        src[2] = decodeLir((ImList)stmt.elem4th());
        return newLir.operator(code, Type.UNKNOWN, src);
      }
        
    case Op.CALL: // currently with type
      {
        int type = Type.decode((String)stmt.elem2nd());
        LirNode callee = decodeLir((ImList)stmt.elem3rd());
        int n = ((ImList)stmt.elem4th()).length(); // parameter list length
        LirNode[] param = new LirNode[n];
        int i = 0;
        for (ImList p = (ImList)stmt.elem4th(); !p.atEnd(); p = p.next())
          param[i++] = decodeLir((ImList)p.elem());
        return newLir.operator(code, type, callee,
                               newLir.operator(Op.LIST, Type.UNKNOWN, param));
      }

    case Op.JUMPN:
      {
        LirNode value = decodeLir((ImList)stmt.elem2nd());
        ImList cases = (ImList)stmt.elem3rd();
        int n = cases.length();
        LirNode[] labels = new LirNode[n];
        int i = 0;
        for (ImList p = cases; !p.atEnd(); p = p.next()) {
          ImList c = (ImList)p.elem();
          labels[i++] = newLir.operator(Op.LIST, Type.UNKNOWN,
                                        decodeLir((ImList)c.elem()),
                                        decodeLir((ImList)c.elem2nd()));
        }
        return newLir.operator(code, Type.UNKNOWN, value,
                               newLir.operator(Op.LIST, Type.UNKNOWN, labels),
                               decodeLir((ImList)stmt.elem4th()));
      }
        
    case Op.PROLOGUE:
    case Op.EPILOGUE:
      {
        int n = stmt.next().length();
        LirNode[] opr = new LirNode[n];
        ImList frame = (ImList)stmt.elem2nd();
        opr[0] = newLir.operator(Op.LIST, Type.UNKNOWN,
                                 newLir.iconst(Type.ADDRESS,
                                               Integer.parseInt((String)frame.elem())),
                                 newLir.iconst(Type.ADDRESS,
                                               Integer.parseInt((String)frame.elem2nd())));
        int i = 1;
        for (ImList p = stmt.next().next(); !p.atEnd(); p = p.next())
          opr[i++] = decodeLir((ImList)p.elem());
        return newLir.operator(code, Type.UNKNOWN, opr);
      }

    case Op.PARALLEL:
      {
        int n = stmt.next().length();
        LirNode[] src = new LirNode[n];
        int i = 0;
        for (ImList p = stmt.next(); !p.atEnd(); p = p.next())
          src[i++] = decodeLir((ImList)p.elem());
        return newLir.operator(code, Type.UNKNOWN, src);
      }

    default:
      throw new Error("Unknown opCode");
    }

  }

  /** Purge former analysis */
  public void purgeAnalysis() {
    analyses.clear();
  }

  /** Apply some analysis */
  public LocalAnalysis apply(LocalAnalyzer analyzer) {
    LocalAnalysis analysis = analyzer.doIt(this);
    analyses.put(analyzer, analysis);
    return analysis;
  }

  /** Require analysis. */
  public LocalAnalysis require(LocalAnalyzer analyzer) {
    LocalAnalysis ana = (LocalAnalysis)analyses.get(analyzer);
    if (ana == null || !ana.isUpToDate())
      ana = apply(analyzer);
    return ana;
  }

  /** Apply some transformation/optimization. */
  public void apply(LocalTransform transformer) {
    transformer.doIt(this);
  }

  /** Code generation (including machine-dependent optimization) */
  public void generateCode(PrintWriter codeFile) {
    // Pass this function to InstSel module
    //  (currently implemented by another program)

    String command = "/bin/cat";
    try {
      Process proc = Runtime.getRuntime().exec(command);
      PrintWriter out = new PrintWriter(proc.getOutputStream());
      Thread recv = new InstSelReceiver(proc.getInputStream(), this);
      recv.start();
      printStandardForm(out);
      out.close();

      int status;
      for (;;) {
        try {
          status = proc.waitFor();
          recv.join();
          break;
        } catch (InterruptedException e) {
          // retry
        }
      }
      proc.destroy();
    } catch (IOException e) {
      throw new Error("can't exec " + command);
    }

    // try {
    //   PrintWriter out = new PrintWriter(new FileOutputStream("functions", true));
    //    printStandardForm(out);
    //    out.close();
    //  } catch(FileNotFoundException e) {
    //    throw new Error("can't create file: functions");
    //  }
  }

  /** Receive output of InstSel process */
  public void receiveInstSel(InputStream in) {
    PushbackReader reader = new PushbackReader(new InputStreamReader(in));
    try {
      Object sexp = ImList.readSexp(reader);
      debOut.println("Returned function body:");
      debOut.println(sexp.toString());
    } catch (IOException e) {
      throw new Error(e.getMessage());
    } catch (SyntaxError e) {
      throw new Error("Syntax error: " + e.getMessage());
    }
  }

  /** Print L-function in standard form. */
  public void printStandardForm(PrintWriter out) {
    out.println("(FUNCTION \"" + symbol.name + "\"");
    // print symbol table
    localSymtab.printStandardForm(out, "  ");
    // print body
    flowGraph.printStandardForm(out, "  ");
    out.println(")");
  }


  /** Dump internal data structure of the Function object. */
  public void printIt(PrintWriter out) { printIt(out, null); }


  /** Dump internal data structure of the Function with some analyses. */
  public void printIt(PrintWriter out, LocalAnalyzer[] anals) {
    out.println();
    out.println("Function \"" + symbol.name + "\":");
    out.print(" Local ");
    localSymtab.printIt(out);
    out.println();
    out.println("Control Flow Graph:");
    flowGraph.printIt(out);

    if (anals != null) {
      for (int i = 0; i < anals.length; i++)
        require(anals[i]).printIt(out);
    }
  }

}
