package coins.backend.cfg;

import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.lir.*;
import coins.backend.sym.*;
import coins.backend.util.*;

/** Represent Control Flow Graph, a directed graph whose
 *   nodes are basic blocks. */
public class FlowGraph {
  /** Function owning this CFG. */
  public final Function function;

  /** List of basic blocks. */
  public final BiList basicBlkList = new BiList();

  /** Identifier of the next allocated block */
  private int blkCounter = 1;

  /** Maximum number of DFN */
  private int maxDfn;

  /** Last modified time but not real time-of-the-day. Incremented
   * when the graph touched. */
  private int timeStamp = 0;

  /** Time stamp of DFST computation. */
  private int dfstTimeStamp = 0;

  /** Split LIR instruction list into basic blocks
   *   and build up CFG. */
  public FlowGraph(Function f, BiList anInstrList) {
    function = f;

    // Partition instructions into basic blocks.
    boolean prevJump = false;
    boolean prevLabel = false;
    BiLink left;
    for (BiLink ptr = anInstrList.first(); !ptr.atEnd(); ptr = left) {
      left = ptr.next();
      LirNode node = (LirNode)ptr.elem();
      node.setRoot(true);
      if (prevJump || node.opCode == Op.DEFLABEL && !prevLabel) {
        BiList fragment = anInstrList;
        anInstrList = anInstrList.split(ptr);
        if (!prevJump) {
          // add explicit JUMP instruction goes to next block
          
          fragment.add(function.newLir.operator(Op.JUMP, Type.UNKNOWN,
                        function.newLir.labelRef(Op.LABEL, Type.ADDRESS,
                                                 ((LirLabelRef)node).label)));
        }
        basicBlkList.add(newBasicBlk(fragment));
      }
      switch (node.opCode) {
      case Op.DEFLABEL:
        prevLabel = true;
        prevJump = false;
        break;
      case Op.JUMP:
      case Op.JUMPC:
      case Op.JUMPN:
        prevJump = true;
        prevLabel = false;
        break;
      default:
        prevLabel = prevJump = false;
        break;
      }
    }
    if (!anInstrList.isEmpty()) {
      // last fragment
      basicBlkList.add(newBasicBlk(anInstrList));
    }

    // Add edges
    for (BiLink bp = basicBlkList.first(); !bp.atEnd(); bp = bp.next()) {
      BasicBlk blk = (BasicBlk)bp.elem();
      blk.maintEdges();
    }

    touch();
    dfstOrder();
  }


  /** Return last modified time of the graph. Time is just a counter
      rather than actual time-of-the-day. */
  public int timeStamp() { return timeStamp; }

  /** Notify that the graph has been modified. */
  public void touch() {
    timeStamp++;
  }

  /** Create new basic block with instruction list <code>instr</code>. */
  BasicBlk newBasicBlk(BiList instr) {
    return new BasicBlk(this, blkCounter++, instr);
  }


  /** Insert new empty basic block before block x. */
  public BasicBlk insertNewBlkBefore(BasicBlk x) {
    BiList list = new BiList();
    BasicBlk blk = newBasicBlk(list);
    list.add(function.newLir.operator(Op.JUMP, Type.UNKNOWN,
                 function.newLir.labelRef(Op.LABEL, Type.ADDRESS, x.label())));
    basicBlkList.locate(x).addBefore(blk);
    blk.maintEdges();
    return blk;
  }


  /** Return maximum block numer + 1. */
  public int idBound() { return blkCounter; }
  

  /** Make Depth First Spanning Tree. Fields dfn(Depth First Number in
   *  rever postorder), dfnPre(preorder), parent are set. */
  public void dfstOrder() { dfstOrderHook(null); }

  /** Depth First Ordering */
  public void dfstOrderHook(DfstHook h) {
    // Has DFST been made already?
    if (dfstTimeStamp >= timeStamp)
      return;

    dfstTimeStamp = timeStamp;

    for (BiLink p = basicBlkList.first(); !p.atEnd(); p = p.next()) {
      BasicBlk x = (BasicBlk)p.elem();
      x.dfn = x.dfnPre = 0;
      x.parent = null;
    }

    int[] cpre = new int[1];
    int[] crpost = new int[1];
    entryBlk().depthFirstSearch(h, null, cpre, crpost);
    maxDfn = crpost[0];

    for (BiLink p = basicBlkList.first(); !p.atEnd(); p = p.next()) {
      BasicBlk x = (BasicBlk)p.elem();
      if (x.dfn != 0)
        x.dfn = crpost[0] - x.dfn + 1;
    }
  }

  /** Return maximum number of DFN (depth first number). */
  public int maxDfn() { return maxDfn; }

  /** Return the vector of basic block indexed by DFN reverse
   * postorder. Zeroth element is null, entry block is in [1]. */
  public BasicBlk[] blkVectorByRPost() {
    dfstOrder();
    BasicBlk[] vec = new BasicBlk[maxDfn + 1];
    for (BiLink p = basicBlkList.first(); !p.atEnd(); p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();
      if (blk.dfn != 0)
        vec[blk.dfn] = blk;
    }
    return vec;
  }

  /** Return the vector of basic block indexed by DFN preorder.
   * Zeroth element is null, entry block is in [1]. */
  public BasicBlk[] blkVectorByPre() {
    dfstOrder();
    BasicBlk[] vec = new BasicBlk[maxDfn + 1];
    for (BiLink p = basicBlkList.first(); !p.atEnd(); p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();
      if (blk.dfnPre != 0)
        vec[blk.dfnPre] = blk;
    }
    return vec;
  }

  /** Return the entry basic block. */
  public BasicBlk entryBlk() { return (BasicBlk)basicBlkList.first().elem(); }

  /** Return the list of basic blocks. */
  // public BiList basicBlkList() { return basicBlkList; }

  /** Return iterator for accessing basic blocks. */
  public Iterator basicBlkIterator() { return basicBlkList.iterator(); }

  /** Print standard form */
  public void printStandardForm(PrintWriter output, String indent) {
    for (BiLink bp = basicBlkList.first(); !bp.atEnd(); bp = bp.next()) {
      BasicBlk blk = (BasicBlk)bp.elem();
      blk.printStandardForm(output, indent);
    }
  }

  /** Print CFG */
  public void printIt(PrintWriter output) {
    dfstOrder();
    for (BiLink bp = basicBlkList.first(); !bp.atEnd(); bp = bp.next()) {
      BasicBlk blk = (BasicBlk)bp.elem();
      blk.printIt(output);
    }
  }

}
