package coins.backend.cfg;

import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.cfg.*;
import coins.backend.util.*;

/** Find dominators of the control flow graph. */
public class Dominators {

  /** Father node in the dominator tree */
  public final int [] idom;
  /** Children node */
  public final BasicBlk [][] kids;

  private BasicBlk[] blkvrpn;
  private int maxRpn;

  /** Create dominator tree for Graph g. */
  public Dominators(FlowGraph g) {
    // Max rpost num
    int maxRpn = 0;
    for (BiLink p = g.basicBlkList.first(); !p.atEnd(); p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();
      if (blk.rpostNum != 0) {
        if (blk.rpostNum > maxRpn)
          maxRpn = blk.rpostNum;
      }
    }
    // allocate idom
    idom = new int[maxRpn + 1];
    blkvrpn = new BasicBlk[maxRpn + 1];
    for (BiLink p = g.basicBlkList.first(); !p.atEnd(); p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();
      if (blk.rpostNum != 0) {
        blkvrpn[blk.rpostNum] = blk;
        if (blk.parent != null)
          idom[blk.rpostNum] = blk.parent.rpostNum;
      }
    }

    // Compute Dominators
    boolean change;
    do {
      change = false;
      for (int i = 2; i <= maxRpn; i++) {
        for (BiLink p = blkvrpn[i].predList().first(); !p.atEnd();
             p = p.next()) {
          int x = idom[i];
          int y = ((BasicBlk)p.elem()).rpostNum;
          while (x != y) {
            if (x > y)
              x = idom[x];
            else /* if (y > x) */
              y = idom[y];
          }
          if (x != idom[i] && x != 0) {
            idom[i] = x;
            change = true;
          }
        }
      }
    } while (change);

    // Now idom represents dominator tree.

    // Set up kids vector
    kids = new BasicBlk[maxRpn + 1][];
    int [] nkids = new int[maxRpn + 1];
    for (int i = 1; i <= maxRpn; i++) {
      if (idom[i] != 0)
        nkids[idom[i]]++;
    }
    for (int i = 1; i <= maxRpn; i++)
      kids[i] = new BasicBlk[nkids[i]];
    for (int i = maxRpn; i >= 1; i--) {
      int j = idom[i];
      if (j != 0)
        kids[j][--nkids[j]] = blkvrpn[i];
    }
  }

  /** Return immediate dominator of block blk. */
  public BasicBlk immDominator(BasicBlk blk) {
    return blkvrpn[idom[blk.rpostNum]];
  }

  /** Return children (immediate dominatees) list iterator of block blk. */
  public Iterator children(BasicBlk blk) {
    class ChildrenIterator implements Iterator {
      int i, j;

      ChildrenIterator(int blkrpn) {
        i = blkrpn;
        j = 0;
      }
      public boolean hasNext() { return j < kids[i].length; }
      public Object next() { return kids[i][j++]; }
      public void remove() { throw new UnsupportedOperationException(); }
    }
    return new ChildrenIterator(blk.rpostNum);
  }

  /** Dump dominator tree on stream out */
  public void printIt(PrintWriter out) {
    out.println();
    out.println("Dominator Tree:");
    dumpIt(1, "", "", out);
  }

  private void dumpIt(int blkrpn, String pref1, String pref2, PrintWriter out) {
    out.print(pref1);
    if (pref2.length() > 0)
      out.print(" +-");
    out.print("#" + blkvrpn[blkrpn].ident + "'s children: ");
    boolean first = true;
    int n = kids[blkrpn].length;
    for (int i = 0; i < n; i++) {
      out.print(first ? "#" : ",#");
      out.print(kids[blkrpn][i].ident);
      first = false;
    }
    out.println();
    String pref = pref1 + pref2;
    for (int i = 0; i < n; i++)
      dumpIt(kids[blkrpn][i].rpostNum, pref,
             (i + 1 == n) ? "   " : " | ", out);
  }
}
