package coins.backend.ana;

import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.cfg.*;
import coins.backend.lir.*;
import coins.backend.sym.*;
import coins.backend.util.*;

/** Live variable analysis.
 *  Set information is represented by bitmap vectors. */
public class LiveVariableAnalysis extends DataFlowAnalysis {

  /** Factory class of LiveVariableAnalysis. */
  private static class Analyzer implements LocalAnalyzer {
    public LocalAnalysis doIt(Function func) {
      return new LiveVariableAnalysis(func);
    }
  }


  /** Factory singleton. */
  public static final Analyzer analyzer = new Analyzer();


  /** Live information for each basic block **/
  private BitSet [] in;
  private BitSet [] out;
  private BitSet [] gen;
  private BitSet [] kill;

  private BitSet [] liveAt;

  private int maxSyms;
  private Symbol[] symVec;


  /** Construct live variable information */
  public LiveVariableAnalysis(Function f) {
    super(f);
  }


  
  /* Interface of Live Variable Analysis */

  /** Return true if variable x is live at position (blk, stmt). **/
  public boolean isLiveAt(Symbol regvar, BasicBlk blk, LirNode stmt) {
    return liveAt[stmt.id].get(regvar.id);
  }

  

  /* Problem-Oriented Methods called by super class (DataFlowAnalysis). */

  /** Initialize problem-oriented data structure. **/
  void initialize() {
    isForward = false; // live variable anal. is backward-direction problem

    int maxBlks = function.flowGraph.idBound();
    maxSyms = function.localSymtab.idBound();
    symVec = function.localSymtab.sortedSymbols();
    in = new BitSet[maxBlks];
    out = new BitSet[maxBlks];
    kill = new BitSet[maxBlks];
    gen = new BitSet[maxBlks];
    for (int i = 0; i < maxBlks; i++) {
      in[i] = new BitSet(maxSyms);
      out[i] = new BitSet(maxSyms);
    }

    
    // Compute kill and gen.
    for (BiLink p = function.flowGraph.basicBlkList.first(); !p.atEnd();
         p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();
      kill[blk.id] = new BitSet(maxSyms);
      gen[blk.id] = new BitSet(maxSyms);

      // Process implicit assignments
      doPhi(blk, kill[blk.id], gen[blk.id]);

      // Scan LirNodes list backward
      for (BiLink q = blk.instrList().last(); !q.atEnd(); q = q.prev()) {
        LirNode stmt = (LirNode)q.elem();

        doStmt(stmt, kill[blk.id], gen[blk.id]);
      }
    }

  }


  /** Transfer kill and gen set over implicit assignments
   *  hiden in PHI nodes located in successors of blk. */
  private void doPhi(BasicBlk blk, BitSet kill, BitSet gen) {
    for (BiLink s = blk.succList().first(); !s.atEnd(); s = s.next()) {
      BasicBlk succ = (BasicBlk)s.elem();

      for (BiLink q = succ.instrList().first(); !q.atEnd(); q = q.next()) {
        LirNode stmt = (LirNode)q.elem();
        if (stmt.opCode != Op.PHI)
          break;

        // LHS of PHI
        doDef(kill, gen, stmt.src(0));

        // Scan RHS of PHI seeking for variables belongs to this block.
        int n = stmt.nSrcs();
        for (int i = 1; i < n; i++) {
          if (stmt.src(i).src(1).opCode == Op.LABEL
              && ((LirLabelRef)stmt.src(i).src(1)).label == blk.label())
            doRef(kill, gen, stmt.src(i).src(0));
        }
      }
    }
  }


  /** Transfer kill and gen set over a statement. **/
  private void doStmt(LirNode stmt, BitSet kill, BitSet gen) {
    switch (stmt.opCode) {
    case Op.PARALLEL:
      {
        int n = stmt.nSrcs();
        // Process all definitions first.
        for (int i = 0; i < n; i++) {
          switch(stmt.src(i).opCode) {
          case Op.SET:
            doDef(kill, gen, stmt.src(i).src(0));
            break;
          case Op.CLOBBER:
            doDef(kill, gen, stmt.src(i).src(0));
            break;
          case Op.USE:
            // do nothing
            break;
          }
        }
        
        // Process references.
        for (int i = 0; i < n; i++) {
          switch(stmt.src(i).opCode) {
          case Op.SET:
            if (stmt.src(i).src(0).opCode == Op.MEM)
              collectRefs(kill, gen, stmt.src(i).src(0));
            collectRefs(kill, gen, stmt.src(i).src(1));
            break;
          case Op.CLOBBER:
            // do nothing
            break;
          case Op.USE:
            doRef(kill, gen, stmt.src(i).src(0));
            break;
          }
        }
      }
      break;
      
    case Op.SET:
      if (stmt.src(0).opCode == Op.MEM)
        collectRefs(kill, gen, stmt.src(0));
      else
        doDef(kill, gen, stmt.src(0));
      collectRefs(kill, gen, stmt.src(1));
      break;
      
    case Op.CLOBBER:
      doDef(kill, gen, stmt.src(0));
      break;
      
    case Op.PROLOGUE:
      {
        int n = stmt.nSrcs();
        for (int i = 1; i < n; i++)
          doDef(kill, gen, stmt.src(i));
      }
      break;
      
      // case Op.CALL:
      
    case Op.PHI:
      // Do nothing - PHI is dealt at predecessor blocks.
      break;
      
    default:
      collectRefs(kill, gen, stmt);
      break;
    }
  }


  /** Look for register variable references and make up KILL/GEN set. **/
  private void collectRefs(BitSet kill, BitSet gen, LirNode node) {
    doRef(kill, gen, node);
    if (node.opCode != Op.SUBREG) {
      int n = node.nSrcs();
      for (int i = 0; i < n; i++)
        collectRefs(kill, gen, node.src(i));
    }
  }

  /** If the node is register variable, treat it as referenced **/
  private void doRef(BitSet kill, BitSet gen, LirNode node) {
    if (node.opCode == Op.REG) {
      int refd = ((LirSymRef)node).symbol.id;
      gen.set(refd);
      if (kill != null) kill.set(refd, false);
    } else if (node.opCode == Op.SUBREG && node.src(0).opCode == Op.REG) {
      doRef(kill, gen, node.src(0));
    }
  }

  /** If the node is register variable, treat it as defined **/
  private void doDef(BitSet kill, BitSet gen, LirNode node) {
    if (node.opCode == Op.REG) {
      int defd = ((LirSymRef)node).symbol.id;
      if (kill != null) kill.set(defd);
      gen.set(defd, false);
    } else if (node.opCode == Op.SUBREG && node.src(0).opCode == Op.REG) {
      doDef(kill, gen, node.src(0));
    }
  }



  /** Supply confluence operator (=join at here) for block blk. **/
  void confluence(BasicBlk blk) {
    out[blk.id].clear();
    for (BiLink q = blk.succList().first(); !q.atEnd(); q = q.next()) {
      BasicBlk b = (BasicBlk)q.elem();
      out[blk.id].or(in[b.id]);
    }
  }

  /** Supply transfer function for block blk. */
  boolean transfer(BasicBlk blk) {
    BitSet newin = (BitSet)out[blk.id].clone();
    newin.andNot(kill[blk.id]);
    newin.or(gen[blk.id]);
    boolean changed = !newin.equals(in[blk.id]);
    in[blk.id] = newin;
    return changed;
  }


  /** Finalization. Set up local-live information. **/
  void windUp() {
    liveAt = new BitSet[function.newLir.idBound()];

    for (BiLink p = function.flowGraph.basicBlkList.first(); !p.atEnd();
         p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();
      BitSet live = (BitSet)out[blk.id].clone();

      doPhi(blk, null, live);

      for (BiLink q = blk.instrList().last(); !q.atEnd(); q = q.prev()) {
        LirNode stmt = (LirNode)q.elem();
        liveAt[stmt.id] = (BitSet)live.clone();

        doStmt(stmt, null, live);
      }
    }
  }

  /** Print Live variables */
  private void printLive(String head, BitSet live, PrintWriter output) {
    output.print(head);
    for (int i = 0; (i = live.nextSetBit(i)) >= 0; i++)
      output.print(" " + symVec[i].name);
    output.println();
  }

  /** Print internal state for debugging (OBSOLETED). **/
  public void printIt(PrintWriter out) {  }

  /** Debug print entries required by interface. **/

  public void printBeforeFunction(PrintWriter output) {}

  public void printBeforeBlock(BasicBlk blk, PrintWriter output) {
    printLive("   kill:", kill[blk.id], output);
    printLive("    gen:", gen[blk.id], output);
    printLive("    in:", in[blk.id], output);
  }

  public void printAfterBlock(BasicBlk blk, PrintWriter output) {
    printLive("   out:", out[blk.id], output);
  }

  public void printBeforeStmt(LirNode stmt, PrintWriter output) {}

  public void printAfterStmt(LirNode stmt, PrintWriter output) {
    printLive("  live:", liveAt[stmt.id], output);
  }

  public void printAfterFunction(PrintWriter output) {}

}
