package coins.backend.sym;

import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.util.*;

/** Symbol table. */
public class SymTab {
  private Map table = new HashMap();
  private BiList list = new BiList();

  /** Next symbol's id */
  private int idCounter = 1;

  /** Create empty symbol table. */
  public SymTab() {}

  /** Register STATIC symbol entry. */
  public Symbol addSymbol(String name, int storage, int type,
                        int boundary, String segment, String linkage) {
    Symbol sym = new SymStatic(name, idCounter++,
                               storage, type, boundary, segment, linkage);
    table.put(name, sym);
    list.add(sym);
    return sym;
  }

  /** Register FRAME/REG symbol entry. */
  public Symbol addSymbol(String name, int storage, int type,
                        int boundary, int offset) {
    Symbol sym = new SymAuto(name, idCounter++,
                             storage, type, boundary, offset);
    table.put(name.intern(), sym);
    list.add(sym);
    return sym;
  }

  /** Find a symbol with key <code>name</code>. */
  public Symbol get(String name) { return (Symbol)table.get(name.intern()); }

  /** Return the list of symbol entries. */
  public BiList symbols() { return list; }

  /** Return an iterator for accessing symbol entries. */
  public Iterator iterator() { return list.iterator(); }

  /** Return array of symbols sorted by their id numbers. */
  public Symbol[] sortedSymbols() {
    Symbol[] vec = new Symbol[idCounter];
    for (BiLink p = list.first(); !p.atEnd(); p = p.next()) {
      Symbol sym = (Symbol)p.elem();
      vec[sym.id] = sym;
    }
    return vec;
  }

  /** Return max id of the symbol + 1. */
  public int idBound() { return idCounter; }

  /** Print symbol table in standard form */
  public void printStandardForm(PrintWriter out, String indent) {
    out.println(indent + "(SYMTAB");
    for (BiLink p = list.first(); !p.atEnd(); p = p.next())
      out.println(indent + "  " + ((Symbol)p.elem()).toString());
    out.println(indent + ")");
  }

  /** Dump symbol table */
  public void printIt(PrintWriter out) {
    out.println("Symbol table:");
    for (BiLink p = list.first(); !p.atEnd(); p = p.next()) {
      out.println("  " + ((Symbol)p.elem()).toString());
    }
  }
}
