package coins.backend.cfg;

import java.io.*;
import coins.backend.*;
import coins.backend.sym.*;
import coins.backend.lir.*;
import coins.backend.util.*;

/** Represent basic block, a sequence of LIR instructions
 * without intervening JUMPs. */
public class BasicBlk {
  /** FlowGraph owning this block. */
  public final FlowGraph flowGraph;

  /** Identifier of this block. */
  public final int id;

  /** Instructions list */
  private BiList instrList;

  /** Successors of this block. */
  private BiList succList = new BiList();

  /** Predecessors of this basic block. */
  private BiList predList = new BiList();

  /** DFST Reverse postorder number */
  int dfn;

  /** DFST Preorder number */
  int dfnPre;

  /** Parent node in the DFST. */
  BasicBlk parent;

  /** This block's label */
  private Label label;

  /** Create basic block with the fragment of instruction list.
   *  Called only by FlowGraph. */
  BasicBlk(FlowGraph g, int idnum, BiList instr) {
    flowGraph = g;
    id = idnum;
    instrList = instr;

    // Have labels point to this block
    for (BiLink insp = instrList.first(); !insp.atEnd();) {
      LirNode node = (LirNode)insp.elem();
      BiLink next = insp.next();
      if (node.opCode == Op.DEFLABEL) {
        ((LirLabelRef)node).label.setBasicBlk(this);
        if (label == null)
          label = ((LirLabelRef)node).label;
        // Delete DEFLABEL ops.
        insp.unlink();
      }
      insp = next;
    }
    // If no labels are shown, add arbitrary label
    if (label == null) {
      String name = (".blk" + id).intern();
      Label label = new Label(name);
      g.function.labelTable.put(name, label);
      label.setBasicBlk(this);
      setLabel(label);
    }
  }

  /** Return instruction list */
  public BiList instrList() { return instrList; }

  /** Replace instruction list */
  void setInstrList(BiList list) { instrList = list; }

  /** Return label of this block */
  public Label label() { return label; }

  /** Replace label */
  public void setLabel(Label l) { label = l; }

  /** Return list of successors */
  public BiList succList() { return succList; }

  /** Return list of predecessors */
  public BiList predList() { return predList; }

  /** Return reverse post order number in the DFST. */
  public int dfn() { return dfn; }

  /** Return preorder number in the DFST. */
  public int dfnPre() { return dfnPre; }

  /** Return parent node in the DFST. */
  public BasicBlk parent() { return parent; }

  /** Add an edge from this block to toBlk */
  void addEdge(BasicBlk toBlk) {
    succList.addNew(toBlk);
    toBlk.predList.addNew(this);
  }

  /** Remove the edge from this block to toBlk */
  void removeEdge(BasicBlk toBlk) {
    succList.remove(toBlk);
    toBlk.predList.remove(toBlk);
  }

  /** Remove all edges from this block */
  public void clearEdges() {
    for (BiLink p = succList.first(); !p.atEnd(); ) {
      BasicBlk target = (BasicBlk)p.elem();
      p = p.next();
      target.predList.remove(this);
    }
    succList.clear();
  }

  /** Maintain edges from this block.
   * Automatically checks JUMP(/C/N) instructions  */
  public void maintEdges() {
    flowGraph.touch();
    clearEdges();

    // check last instruction
    if (!instrList.isEmpty()) {
      Label[] targets = ((LirNode)instrList.last().elem()).getTargets();
      if (targets != null) {
        for (int i = 0; i < targets.length; i++)
          addEdge(targets[i].basicBlk());
      }
    }
  }

  /** Depth First Search */
  void depthFirstSearch(DfstHook h, BasicBlk from, int[] cpre, int[] crpost) {
    if (h != null)
      h.preOrder(this, from);
    dfnPre = ++cpre[0];
    parent = from;
    for (BiLink p = succList.first(); !p.atEnd(); p = p.next()) {
      BasicBlk y = (BasicBlk)p.elem();
      if (y.dfnPre == 0)
        y.depthFirstSearch(h, this, cpre, crpost);
    }
    if (h != null)
      h.postOrder(this);
    dfn = ++crpost[0];
  }

  /** Print this block */
  public void printIt(PrintWriter output) {
    output.println();
    output.print("  #" + id + " Basic Block");
    if (label != null)
      output.print(" (" + label + ")");
    output.print(": DFN=(" + dfnPre + "," + dfn + "),");
    if (parent != null)
      output.print(" parent=#" + parent.id + ",");
    output.print(" <-(");
    boolean middle = false;
    for (BiLink blkp = predList.first(); !blkp.atEnd(); blkp = blkp.next()) {
      BasicBlk blk = (BasicBlk)blkp.elem();
      output.print((middle ? ",#" : "#") + blk.id);
      middle = true;
    }
    output.print(") ->(");
    middle = false;
    for (BiLink blkp = succList.first(); !blkp.atEnd(); blkp = blkp.next()) {
      BasicBlk blk = (BasicBlk)blkp.elem();
      output.print((middle ? ",#" : "#") + blk.id);
      middle = true;
    }
    output.println(")");
    for (BiLink insp = instrList.first(); !insp.atEnd(); insp = insp.next()) {
      LirNode node = (LirNode)insp.elem();
      output.print("    ");
      output.println(node.toString());
    }
  }

}
