package coins.backend.lir;

import coins.backend.*;
import coins.backend.sym.*;
import coins.backend.cfg.*;

/**
 * New LIR node's basic structure
 *  
 */
public abstract class LirNode {
  /** Key to bind related information */
  public final int id;

  /** Instruction Code */
  public final int opCode;

  /** Type of this node's result */
  public final int type;

  /** Miscellaneous attributes */
  int flags;
  static final int F_ROOT = 1;
  
  /** Create LIR node */
  LirNode(int id, int opCode, int type) {
    this.id = id;
    this.opCode = opCode;
    this.type = type;
  }

  /** Return number of operands */
  public int nSrcs() { return 0; }

  /** Return nth operand; Subclass responsibility */
  public LirNode src(int n) { throw new IllegalArgumentException(); }

  /** Set nth operand; Subclass responsibility */
  public void setSrc(int n, LirNode src) {
    throw new IllegalArgumentException();
  }

  /** Return jump target labels. Targets are packed in an array of Label. */
  public Label[] getTargets() { return null; }

  /** Return true if this node is the root of the tree. */
  public boolean isRoot() { return (flags & F_ROOT) != 0; }

  /** Set true/false on root flag */
  public void setRoot(boolean yes) {
    if (yes)
      flags |= F_ROOT;
    else
      flags &= ~F_ROOT;
  }

  /** Visualize */
  public String toString() {
    StringBuffer buf = new StringBuffer();
    //    if (id != 0)
    //      buf.append("$" + id + ": ");
    buf.append("(");
    if (opCode != Op.LIST) {
      buf.append(Op.toName(opCode));
      if (type != Type.UNKNOWN || opCode == Op.CALL) {
        buf.append(Debug.TypePrefix);
        buf.append(Type.toString(type));
      }
      buf.append(" ");
    }
    int n = nSrcs();
    for (int i = 0; i < n; i++) {
      if (i != 0)
        buf.append(" ");
      buf.append(src(i).toString());
    }
    buf.append(")");
    return buf.toString();
  }

  /** Accept visitor v */
  public abstract void accept(LirVisitor v);

}
