package coins.backend;

import java.lang.*;
import java.io.*;
import coins.backend.*;
import coins.backend.ana.*;
import coins.backend.opt.*;
import coins.backend.util.*;

/**
 * Back end of COINS compiler.
 */
public class BackEnd {

  /** Stream to print debugging information. **/
  public final PrintWriter debOut;

  /** Flag indicating whether or not generate code. **/
  public final boolean doCodeGen;

  /** Flag indicating whether or not dump S-expression just after read. **/
  public final boolean dumpAfterRead = true;

  /** Flag indicating whether or not dump Module built. **/
  public final boolean dumpModule = true;


  private static final String TMDFILE = "test.tmd";


  public BackEnd(boolean codeGen, PrintWriter debOut) {
    this.doCodeGen = codeGen;
    this.debOut = debOut;
  }


  public void doIt(ImList sexp, OutputStream codeStream)
    throws SyntaxError, IOException {
    debOut.println(sexp.toString());
    // Read LIR
    Module compileUnit = new Module(sexp, TMDFILE, this);

    // Apply optimization
    compileUnit.apply(new IntroVirReg(this));
    if (dumpModule) {
      debOut.println();
      debOut.println("After Virtual Register Replacement:");
      compileUnit.printIt(debOut);
    }

    // Optimize Jump instructions.
    compileUnit.apply(new JumpOpt(this));

    // Insert loop pre-headers.
    compileUnit.apply(new PreHeaders(this));

    if (dumpModule) {
      debOut.println();
      debOut.println("After Jump Optimization, Preheaders Insertion:");
      compileUnit.printIt(debOut,
                          new LocalAnalyzer[]{Dominators.analyzer,
                                              DominanceFrontiers.analyzer,
                                              LoopAnalysis.analyzer});
    }

    if (doCodeGen) {
      // Instruction Selection, Register Allocation & Code Generation
      compileUnit.apply(new JumpCanon(this));
      if (dumpModule) {
        debOut.println();
        debOut.println("After Jump Canonicalization:");
        compileUnit.printIt(debOut);
      }
      compileUnit.generateCode(new PrintWriter(codeStream));
      
    } else {

      compileUnit.apply(new Ssa(this));
      if (dumpModule) {
        debOut.println();
        debOut.println("After Transformation to SSA:");
        compileUnit.printIt(debOut);
        
        debOut.println();
        debOut.println("After Live Variable Analysis");
        compileUnit.printIt(debOut,
                            new LocalAnalyzer[] {
                              LiveVariableAnalysis.analyzer,
                              InterferenceGraph.analyzer});
      }
    }
  }
}
