package coins.backend;

import java.lang.*;
import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.tmd.*;
import coins.backend.sym.*;
import coins.backend.lir.*;
import coins.backend.util.*;

/**
 * Represent module, the whole input of the compiler.
 */
public class Module {
  /** Module name */
  public final String name;

  /** Global symbol table */
  public final SymTab globalSymtab = new SymTab();

  /** List of functions */
  public final BiList functionList = new BiList();

  // /** Global symbol reference node's serial number */
  // private int idCounter = 1;

  /** Collection of global variables **/
  private BackEnd root;

  /** Target Machine **/
  private TargetMachine rule;

  /** Create a module */
  public Module(Object sexp, String tmdfile, BackEnd root)
    throws SyntaxError, IOException {

    this.root = root;

    // Initialize Target Machine
    rule = new TargetMachine(globalSymtab, tmdfile, root);

    if (!(sexp instanceof ImList))
      throw new SyntaxError("Expected (");
    // root.debOut.println("Module:doCodegen() sexp = " + (ImList)sexp);
    ImList ptr = (ImList)sexp;
    if (ptr.elem() != Keyword.MODULE)
      throw new SyntaxError("Expected MODULE");
    if ((ptr = ptr.next()).atEnd())
      throw new SyntaxError("Expected module name");

    name = ((QuotedString)ptr.elem()).body;

    // Parse top-level S-expressions
    while (!(ptr = ptr.next()).atEnd()) {
      ImList top = (ImList)ptr.elem();
      String kw = (String)top.elem();
      if (kw == Keyword.SYMTAB)
        doSymtbl(top);
      else if (kw == Keyword.DATA)
        doData(top);
      else if (kw == Keyword.FUNCTION) {
        Function f = new Function(this, top, root);
        functionList.add(f);
      }
      else
        throw new SyntaxError("Unexpected " + kw);
    }
  }

  private void doSymtbl(ImList ptr) throws SyntaxError {

    // Parse symbol table
    while (!(ptr = ptr.next()).atEnd()) {
      globalSymtab.addSymbol((ImList)ptr.elem());
    }
  }

  private void doData(ImList aTop) throws SyntaxError {}



  /** Apply some analysis on each function */
  public void apply(LocalAnalyzer analyzer) {
    for (BiLink p = functionList.first(); !p.atEnd(); p = p.next())
      ((Function)p.elem()).apply(analyzer);
  }

  /** Require some analysis for each function */
  public void require(LocalAnalyzer analyzer) {
    for (BiLink p = functionList.first(); !p.atEnd(); p = p.next())
      ((Function)p.elem()).require(analyzer);
  }

  /** Do some transform(or optimization) on each function */
  public void apply(LocalTransform transformer) {
    for (BiLink p = functionList.first(); !p.atEnd(); p = p.next())
      ((Function)p.elem()).apply(transformer);
  }

  /** Code generation */
  public void generateCode(PrintWriter codeFile)
   throws SyntaxError, IOException {
    // Generate external variables.

    // Instruction selection, register allocation
    for (BiLink p = functionList.first(); !p.atEnd(); p = p.next()) {
      Function func = (Function)p.elem();
      p.setElem(func.generateCode(rule));
    }

    // asm out
    rule.asmout(this, codeFile);
  }



  /** Print current module status */
  public void printIt(PrintWriter out) { printIt(out, null); }

  /** Print current module status with analyses */
  public void printIt(PrintWriter out, LocalAnalyzer[] anals) {
    out.println("Module \"" + name + "\":");
    out.print("Global ");
    globalSymtab.printIt(out);
    for (BiLink p = functionList.first(); !p.atEnd(); p = p.next()) {
      ((Function)p.elem()).printIt(out, anals);
    }
  }
}
