package coins.backend.ana;

import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.cfg.*;
import coins.backend.lir.*;
import coins.backend.sym.*;
import coins.backend.util.*;


/** Create interference graph and disturbance graph **/
public class InterferenceGraph implements LocalAnalysis {

  /** Factory class of Dominators. */
  private static class Analyzer implements LocalAnalyzer {
    public LocalAnalysis doIt(Function func) {
      return new InterferenceGraph(func);
    }
  }

  /** Factory singleton. */
  public static final Analyzer analyzer = new Analyzer();


  private Function function;
  private int timeStamp;
  private LiveVariableAnalysis liveInfo;

  private BitSet adjMatrix;

  private Symbol[] regvarVec;
  private BiList[] interfereLists;
  private BiList[] disturbLists;

  /** Create interference graph of register variables in function func. **/
  public InterferenceGraph(Function func) {
    function = func;
    timeStamp = func.flowGraph.timeStamp();

    liveInfo = (LiveVariableAnalysis)func.require(LiveVariableAnalysis.analyzer);
    regvarVec = liveInfo.regvarVec();
    int nRegvars = regvarVec.length;
    
    adjMatrix = new BitSet(nRegvars * (nRegvars - 1) / 2);
    interfereLists = new BiList[nRegvars];
    for (int i = 0; i < nRegvars; i++)
      interfereLists[i] = new BiList();
    disturbLists = new BiList[nRegvars];
    for (int i = 0; i < nRegvars; i++)
      disturbLists[i] = new BiList();

    for (BiLink p = func.flowGraph.basicBlkList.first(); !p.atEnd();
         p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();

      final BitSet live = liveInfo.liveOutBitSet(blk);

      DefUseHandler handler = new DefUseHandler() {
          public void defined(LirNode node) {
            if (node.opCode == Op.SUBREG)
              node = node.src(0);
            int var = liveInfo.regvarIndex(((LirSymRef)node).symbol);
            for (int i = 0; (i = live.nextSetBit(i)) >= 0; i++) {
              setInterfere(var, i);
              setDisturb(var, i);
            }
            live.set(var, false);
          }

          public void used(LirNode node) {
            if (node.opCode == Op.SUBREG)
              node = node.src(0);
            int var = liveInfo.regvarIndex(((LirSymRef)node).symbol);
            for (int i = 0; (i = live.nextSetBit(i)) >= 0; i++) {
              setDisturb(var, i);
            }
            live.set(var);
          }
        };

      for (BiLink q = blk.instrList().last(); !q.atEnd(); q = q.prev()) {
        LirNode stmt = (LirNode)q.elem();

        if (stmt.opCode == Op.PHI) {
          handler.defined(stmt.src(0));
        } else {
          stmt.pickupDefUseReg(handler);
          // liveInfo.maintainLiveOverStmt(live, stmt);
        }
      }
    }
  }

  private void setInterfere(int x, int y) {
    if (x != y) {
      if (x < y) { int w = x; x = y; y = w; }
      if (!adjMatrix.get(x * (x - 1) / 2 + y)) {
        adjMatrix.set(x * (x - 1) / 2 + y);
        interfereLists[x].add(regvarVec[y]);
        interfereLists[y].add(regvarVec[x]);
      }
    }
  }

  private void setDisturb(int x, int y) {
    if (x != y)
      disturbLists[y].addNew(regvarVec[x]);
  }

  /** Return true if register x interferes register y.
   *  Names of register variables are index numbers. **/
  public boolean interfere(int x, int y) {
    if (x == y)
      return false;
    if (x < y) { int w = x; x = y; y = w; }
    return adjMatrix.get(x * (x - 1) / 2 + y);
  }

  /** Return true if register x interferes register y **/
  public boolean interfere(Symbol x, Symbol y) {
    return interfere(liveInfo.regvarIndex(x), liveInfo.regvarIndex(y));
  }

  /** Return the list of variables interfering x. **/
  public BiList interfereList(Symbol x) {
    return interfereLists[liveInfo.regvarIndex(x)];
  }

  /** Return the list of variables disturbing x. **/
  public BiList disturbList(Symbol x) {
    return disturbLists[liveInfo.regvarIndex(x)];
  }



  /** Return true if this analysis is up to date. */
  public boolean isUpToDate() {
    return (timeStamp == function.flowGraph.timeStamp());
  }


  /** Debug print entries required by interface. **/

  public void printBeforeFunction(PrintWriter output) {}
  public void printBeforeBlock(BasicBlk blk, PrintWriter output) {}
  public void printAfterBlock(BasicBlk blk, PrintWriter output) {}
  public void printBeforeStmt(LirNode stmt, PrintWriter output) {}
  public void printAfterStmt(LirNode stmt, PrintWriter output) {}

  public void printAfterFunction(PrintWriter output) {
    output.println();
    output.println("Interference Graph:");

    int n = regvarVec.length;
    for (int i = 0; i < n; i++) {
      output.print(regvarVec[i].name + ":");
      for (BiLink p = interfereLists[i].first(); !p.atEnd(); p = p.next()) {
          Symbol x = (Symbol)p.elem();
          output.print(" " + x.name);
      }
      output.println();
    }

    output.println();
    output.println("Disturbance Graph:");

    for (int i = 0; i < n; i++) {
      output.print(regvarVec[i].name + ":");
      for (BiLink p = disturbLists[i].first(); !p.atEnd(); p = p.next()) {
          Symbol x = (Symbol)p.elem();
          output.print(" " + x.name);
      }
      output.println();
    }
  }

  public void printIt(PrintWriter output) {
    printAfterFunction(output);
  }

}
