package coins.backend.ana;

import java.io.*;
import java.util.*;
import coins.backend.*;
import coins.backend.cfg.*;
import coins.backend.lir.*;
import coins.backend.sym.*;
import coins.backend.util.*;

/** Live variable analysis.
 *  Set information is represented by bitmap vectors. */
public class LiveVariableAnalysis extends DataFlowAnalysis {

  /** Factory class of LiveVariableAnalysis. */
  private static class Analyzer implements LocalAnalyzer {
    public LocalAnalysis doIt(Function func) {
      return new LiveVariableAnalysis(func);
    }
  }


  /** Factory singleton. */
  public static final Analyzer analyzer = new Analyzer();


  /** Live information for each basic block **/
  private BitSet [] in;
  private BitSet [] out;
  private BitSet [] gen;
  private BitSet [] kill;

  // private BitSet [] liveAt;

  private int nRegvars;
  private int nPhyRegs;
  private Symbol[] regvarVec;
  private int[] regvarIndex;



  /** Construct live variable information */
  public LiveVariableAnalysis(Function f) {
    super(f);
  }


  /* Interface of Live Variable Analysis */

  /** Array of register variables **/
  public Symbol[] regvarVec() {
    return regvarVec;
  }

  /** Number of physical registers **/
  public int nPhyRegs() { return nPhyRegs; }

  /** Return symbol's index number used in live set **/
  public int regvarIndex(Symbol sym) {
    if (regvarIndex[sym.id] < 0)
      throw new IllegalArgumentException("Non-register variable");
    return regvarIndex[sym.id];
  }



  /** Return true if variable regvar is live at entry of blk. **/
  public boolean isLiveAtEntry(Symbol regvar, BasicBlk blk) {
    return in[blk.id].get(regvarIndex(regvar));
  }

  /** Return true if variable regvar is live at exit of blk. **/
  public boolean isLiveAtExit(Symbol regvar, BasicBlk blk) {
    return out[blk.id].get(regvarIndex(regvar));
  }



  /** Return the list of live variables at exit of basic block blk. **/
  public BiList liveOut(BasicBlk blk) {
    BiList live = new BiList();
    for (int i = 0; (i = out[blk.id].nextSetBit(i)) >= 0; i++)
      live.addNew(regvarVec[i]);
    return live;
  }


  /** Return the list of live variables at entry of basic block blk. **/
  public BiList liveIn(BasicBlk blk) {
    BiList live = new BiList();
    for (int i = 0; (i = in[blk.id].nextSetBit(i)) >= 0; i++)
      live.addNew(regvarVec[i]);
    return live;
  }

  

  /** Return BitSet of live variables at exit of basic block blk. **/
  BitSet liveOutBitSet(BasicBlk blk) {
    return (BitSet)out[blk.id].clone();
  }


  /** Return BitSet of live variables at entry of basic block blk. **/
  BitSet liveInBitSet(BasicBlk blk) {
    return (BitSet)in[blk.id].clone();
  }




  /* Problem-Oriented Methods called by super class (DataFlowAnalysis). */

  /** Initialize problem-oriented data structure. **/
  void initialize() {
    isForward = false; // live variable anal. is backward-direction problem

    int maxBlks = function.flowGraph.idBound();

    int maxSyms = function.localSymtab.idBound();
    nRegvars = 0;
    for (BiLink p = function.module.globalSymtab.symbols().first(); !p.atEnd();
         p = p.next()) {
      if (((Symbol)p.elem()).storage == Storage.REG)
        nRegvars++;
    }
    nPhyRegs = nRegvars;
    for (BiLink p = function.localSymtab.symbols().first(); !p.atEnd();
         p = p.next()) {
      if (((Symbol)p.elem()).storage == Storage.REG)
        nRegvars++;
    }
    regvarVec = new Symbol[nRegvars];
    regvarIndex = new int[maxSyms];
    for (int i = 0; i < maxSyms; i++)
      regvarIndex[i] = -1;
    int n = 0;
    for (BiLink p = function.module.globalSymtab.symbols().first(); !p.atEnd();
         p = p.next()) {
      Symbol sym = (Symbol)p.elem();
      if (sym.storage == Storage.REG) {
        regvarVec[n] = sym;
        regvarIndex[sym.id] = n;
        n++;
      }
    }
    for (BiLink p = function.localSymtab.symbols().first(); !p.atEnd();
         p = p.next()) {
      Symbol sym = (Symbol)p.elem();
      if (sym.storage == Storage.REG) {
        regvarVec[n] = sym;
        regvarIndex[sym.id] = n;
        n++;
      }
    }
    
    in = new BitSet[maxBlks];
    out = new BitSet[maxBlks];
    kill = new BitSet[maxBlks];
    gen = new BitSet[maxBlks];
    for (int i = 0; i < maxBlks; i++) {
      in[i] = new BitSet(nRegvars);
      out[i] = new BitSet(nRegvars);
    }

    // Compute kill and gen.
    for (BiLink p = function.flowGraph.basicBlkList.first(); !p.atEnd();
         p = p.next()) {
      BasicBlk blk = (BasicBlk)p.elem();
      kill[blk.id] = new BitSet(nRegvars);
      gen[blk.id] = new BitSet(nRegvars);

      // Scan LirNodes list backward
      for (BiLink q = blk.instrList().last(); !q.atEnd(); q = q.prev()) {
        LirNode stmt = (LirNode)q.elem();
        if (stmt.opCode != Op.PHI)
          doStmt(stmt, kill[blk.id], gen[blk.id]);
      }
    }

  }



  /** Maintain kill/gen sets over a statement. **/
  private void doStmt(LirNode stmt, final BitSet kill, final BitSet gen) {

    stmt.pickupDefUseReg(new DefUseHandler() {
        public void defined(LirNode node) {
          if (node.opCode == Op.SUBREG)
            node = node.src(0);
          if (node.opCode == Op.REG) {
            int defd = regvarIndex[((LirSymRef)node).symbol.id];
            if (kill != null) kill.set(defd);
            gen.set(defd, false);
          }
        }

        public void used(LirNode node) {
          if (node.opCode == Op.SUBREG)
            node = node.src(0);
          if (node.opCode == Op.REG) {
            int refd = regvarIndex[((LirSymRef)node).symbol.id];
            if (kill != null) kill.set(refd, false);
            gen.set(refd);
          }
        }
      });
  }





  /** Supply confluence operator (=join at here) for block blk. **/
  void confluence(BasicBlk blk) {
    out[blk.id].clear();
    for (BiLink s = blk.succList().first(); !s.atEnd(); s = s.next()) {
      BasicBlk succ = (BasicBlk)s.elem();
      out[blk.id].or(in[succ.id]);

      /** Special treatment of PHI functions. **/
      /** x0 = phi(x1, x2*, x3); --> kill x0, gen x2 only if x0 is live **/

      for (BiLink q = succ.instrList().first(); !q.atEnd(); q = q.next()) {
        LirNode stmt = (LirNode)q.elem();
        if (stmt.opCode != Op.PHI)
          break;

        // Mark RHS 'used' only if LHS of PHI is live.
        if (out[blk.id].get(regvarIndex[((LirSymRef)stmt.src(0)).symbol.id])) {
          // Scan RHS of PHI which belongs to this block.
          int n = stmt.nSrcs();
          for (int i = 1; i < n; i++) {
            if (stmt.src(i).src(1).opCode == Op.LABEL
                && ((LirLabelRef)stmt.src(i).src(1)).label == blk.label()
                && stmt.src(i).src(0).opCode == Op.REG)
              out[blk.id].set(regvarIndex[((LirSymRef)stmt.src(i).src(0)).symbol.id]);
          }
        }

        // Kill LHS of PHI.
        out[blk.id].set(regvarIndex[((LirSymRef)stmt.src(0)).symbol.id], false);
      }
    }
  }

  /** Supply transfer function for block blk. */
  boolean transfer(BasicBlk blk) {
    BitSet newin = (BitSet)out[blk.id].clone();
    newin.andNot(kill[blk.id]);
    newin.or(gen[blk.id]);
    boolean changed = !newin.equals(in[blk.id]);
    in[blk.id] = newin;
    return changed;
  }


  /** Finalization. **/
  void windUp() { }


  /** Print Live variables */
  private void printLive(String head, BitSet live, PrintWriter output) {
    output.print(head);
    for (int i = 0; (i = live.nextSetBit(i)) >= 0; i++)
      output.print(" " + regvarVec[i].name);
    output.println();
  }

  /** Print internal state for debugging (OBSOLETED). **/
  public void printIt(PrintWriter out) {  }

  /** Debug print entries required by interface. **/

  public void printBeforeFunction(PrintWriter output) {}

  public void printBeforeBlock(BasicBlk blk, PrintWriter output) {
    printLive("   kill:", kill[blk.id], output);
    printLive("    gen:", gen[blk.id], output);
    printLive("    in:", in[blk.id], output);
  }

  public void printAfterBlock(BasicBlk blk, PrintWriter output) {
    printLive("   out:", out[blk.id], output);
  }

  public void printBeforeStmt(LirNode stmt, PrintWriter output) {}

  public void printAfterStmt(LirNode stmt, PrintWriter output) {
    //    printLive("  live:", liveAt[stmt.id], output);
  }

  public void printAfterFunction(PrintWriter output) {}

}
