;;; CLIR - C $BIw(B Syntax $B$N(B LIR$B!#!z$^$@L$40@.!#(B
;;; 
;;; (slir->clir exp)  LIR (S-$B<0(B) $B$r(B CLIR ($BJ8;zNs(B) $B$X!#(B
;;; (clir->slir str)  $B$=$N5U!#(B
;;;

;;; $B1i;;;R>pJs!#(B

(define *clir-code-info*
  '(;; atomic typed prec=0 
    ((INTCONST)
     (FLOATCONST)
     (STATIC)
     (FRAME)
     (LABEL)
     (REG)
     (SUBREG))
    ;; unary prec=1
    ((NEG "-")
     (BNOT "~")
     (MEM "*")
     (CONVSX "sx")
     (CONVZX "zx")
     (CONVIT "it")
     (CONVFX "fx")
     (CONVFT "ft")
     (CONVFI "fi")
     (CONVSF "sf")
     (CONVUF "uf"))
    ;; binary prec=2
    ((MUL "*")
     (DIVS "/")
     (DIVU "/u")
     (MODS "%")
     (MODU "%u"))
    ;; binary prec=3
    ((ADD "+")
     (SUB "-"))
    ;; binary prec=4
    ((LSHS "<<")
     (LSHU "<<u")
     (RSHS ">>")
     (RSHU ">>u"))
    ;; binary prec=5
    ((TSTLTS "<")
     (TSTLES "<=")
     (TSTGTS ">")
     (TSTGES ">=")
     (TSTLTU "<u")
     (TSTLEU "<=u")
     (TSTGTU ">u")
     (TSTGEU ">=u"))
    ;; binary prec=6
    ((TSTEQ "==")
     (TSTNE "!="))
    ;; binary prec=7
    ((BAND "&")
     (BOR "|")
     (BXOR "^"))
    ;; binary prec=8
    ((SET "="))
    ))

(define *clir-code-prec-vec*
  (let ((vec (make-vector (length *clir-code-info*)))
	(prec 0))
    (dolist (x *clir-code-info*)
      (vector-set! vec prec (map car x))
      (inc! prec))
    vec))

(define (clir-highest-prec)
  (1- (vector-length *clir-code-prec-vec*)))

(define (clir-codes-of-prec prec)
  (vector-ref *clir-code-prec-vec* prec))

(define *clir-code-prec-alist*
  (let ((alist (make-tconc))
	(prec 0))
    (dolist (x *clir-code-info*)
      (tconc alist (map (lambda (e) (list (car e) prec)) x))
      (inc! prec))
    (apply append (tconc->list alist))))
  

(define *clir-code-cop-alist*
  (apply append *clir-code-info*))

(define (clir-code-cop code)
  (let ((a (assq code *clir-code-cop-alist*)))
    (unless a (error "%s is not in *clir-code-info*" code))
    (cadr (assq code *clir-code-cop-alist*))))

(define (clir-code-prec code)
  (cadr (assq code *clir-code-prec-alist*)))


;;; CLIR $B$N%l%-%7%+%k%"%J%i%$%6!#(B

(define *clir-lex-token* ())
(define *clir-lex-value* ())

(define (test-clir-lex str)
  (let ((sm (open-input-string str)))
    (while (not (eof-object? (peek-char sm)))
      (set! *clir-lex-token* 'unknown)
      (set! *clir-lex-value* ())
      ;;
      (clir-lex sm)
      ;;
      (format 1 "token = %S  value = %S\n"
	      *clir-lex-token*
	      *clir-lex-value* ))))
  
(define (clir-lex-skip-white sm)
  (while (memv (peek-char sm) '(#\space #\tab #\newline))
    (read-char sm)))

(define (clir-lex sm)
  (clir-lex-skip-white sm)
  (let ((c (read-char sm)))
    (set! *clir-lex-token*
	  (case c
	    ((#\") (clir-lex-read-name sm #\"))
	    ((#\') (clir-lex-read-name sm #\'))
	    ((#\`) (clir-lex-read-name sm #\`))
	    ((#\:) 'colon)
	    ((#\+) 'ADD)
	    ((#\-) (if (char-digit? (peek-char sm))
		       (prog1
			   (clir-lex-read-num (read-char sm) sm)
			 (set! *clir-lex-value* (- *clir-lex-value*)))
		     '-))
	    ((#\*) '*)
	    ((#\/) (if (eqv? (peek-char sm) #\u)
		       (begin  (read-char sm) 'DIVU)
		     'DIVS))
	    ((#\%)
	     ;; % %u %sym
	     (let ((c (peek-char sm)))
	       (cond
		((eqv? c #\u) (read-char sm) 'MODU)
		((clir-sym-start-char? c)
		 (clir-lex-read-sym #\% sm))
		(else 'MODS))))
	    ((#\&) 'BAND)
	    ((#\|) 'BOR)
	    ((#\^) 'BXOR)
	    ((#\~) 'BNOT)
	    ((#\<)
	     ;; < <u <= <=u << <<u
	     (let ((c (peek-char sm)))
	       (case c
		((#\u) (read-char sm) 'TSTLTU)
		((#\=)
		 (read-char sm)
		 (let ((c (peek-char sm)))
		   (case c
		     ((#\u) (read-char sm) 'TSTLEU)
		     (else  'TSTLES))))
		((#\<)
		 (read-char sm)
		 (let ((c (peek-char sm)))
		   (case c
		     ((#\u) (read-char sm) 'LSHU)
		     (else 'LSHS))))
		(else 'TSTLTS))))
	    ((#\>)
	     ;; > >u >= >=u >> >>u
	     (let ((c (peek-char sm)))
	      (case c
		((#\u) (read-char sm) 'TSTGTU)
		((#\=)
		 (read-char sm)
		 (let ((c (peek-char sm)))
		   (case c
		     ((#\u) (read-char sm) 'TSTGEU)
		     (else  'TSTGES))))
		((#\>)
		 (read-char sm)
		 (let ((c (peek-char sm)))
		   (case c
		     ((#\u) (read-char sm) 'RSHU)
		     (else 'RSHS))))
		(else 'TSTGTS))))
	    ((#\=)
	     (let ((c (peek-char sm)))
	       (case c
		 ((#\=) (read-char sm) 'TSTEQ)
		 (else 'SET))))
	    ((#\!)
	     (cond
	      ((eqv? (read-char sm) #\=) 'TSTNE)
	      (else
	       (error "Syntax error: expected: !="))))
	    ((#\() 'lpa)
	    ((#\)) 'rpa)
	    ;;
	    (else
	     (cond
	      ((eof-object? c) 'eof)
	      ((clir-sym-start-char? c)
	       (clir-lex-read-sym c sm))
	      ((char-digit? c)
	       (clir-lex-read-num c sm))
	      (else
	       (error "Syntax error: illlegal character: %c" c))))))))


(define (clir-lex-expect sm tk)
  (if (eq? tk *clir-lex-token*)
      (clir-lex sm)
    (error "Syntax error")))

(define (clir-lex-read-name sm termc)
  (let ((s (make-tconc))
	(c ()))
    (call/cc
     (lambda (break)
       (while (not (eqv? (set! c (read-char sm)) termc))
	 (if (eof-object? c)
	     (begin
	       (error "Syntax error: unterminated: %c" termc)
	       (break ()))
	   (tconc s c)))))
    (set! *clir-lex-value* (list->string (tconc->list s)))
    (case termc
      ((#\") 'STATIC)
      ((#\') 'FRAME)
      ((#\`) 'LABEL))))

(define (clir-sym-start-char? c)
  (and (not (eof-object? c))
       (or (char-alphabet? c)
	   (memv c '(#\_ #\% #\.)))))

(define (clir-sym-rest-char? c)
  (and (not (eof-object? c))
       (not (char=? c #\%))
       (or (clir-sym-start-char? c)
	   (char-digit? c))))

(define (clir-lex-read-sym c0 sm)
  (let ((s (make-tconc)))
    (tconc s c0)
    (while (clir-sym-rest-char? (peek-char sm))
      (tconc s (read-char sm)))
    (set! *clir-lex-value* (list->string (tconc->list s)))
    (let ((a (assoc *clir-lex-value*
		    '(("sx" CONVSX)
		      ("zx" CONVZX)
		      ("it" CONVIT)
		      ("fx" CONVFX)
		      ("ft" CONVFT)
		      ("fi" CONVFI)
		      ("sf" CONVSF)
		      ("uf" CONVUF)))))
      (if a (cadr a) 'sym))))

(define (clir-lex-read-num c0 sm)
  (let ((s (make-tconc)))
    (tconc s c0)
    (while (and (not (eof-object? (peek-char sm)))
		(char-digit? (peek-char sm)))
      (tconc s (read-char sm)))
    ;; $B!z(B FLOATCONST $B$^$@!#(B
    (set! *clir-lex-value* (reads (list->string (tconc->list s))))
    'INTCONST))


;;; CLIR $B$N%Q!<%5!#(B

(define (clir->slir str)
  (let ((sm (open-input-string str)))
    (clir-parse-toplevel sm)))

(define (clir-parse-toplevel sm)
  (clir-lex sm)
  (prog1
      (clir-parse-exp sm)
    (clir-lex-expect sm 'eof)))
  

(define (clir-parse-exp sm)
  (clir-parse-binary-exp sm (clir-highest-prec)))

(define (clir-parse-binary-exp sm prec)
  (define (code)
    (case *clir-lex-token*
      ((*) 'MUL)
      ((-) 'SUB)
      (else *clir-lex-token*)))
  (if (<= prec 1)
      (clir-parse-unary-exp sm)
    (let ((bincodes (clir-codes-of-prec prec))
	  (ast ()))
      (dec! prec)
      (set! ast (clir-parse-binary-exp sm prec))
      (while (memq (code) bincodes)
	(let ((code (prog1 (code) (clir-lex sm)))
	      (type (clir-parse-type-annot sm)))
	  (set! ast (list code type ast (clir-parse-binary-exp sm prec)))))
      ast)))
      
(define (clir-parse-unary-exp sm)
  (let ((code (case *clir-lex-token*
		((*) 'MEM)
		((-) 'NEG)
		(else *clir-lex-token*))))
    (cond
     ((memq code (clir-codes-of-prec 1))
      (clir-lex sm)
      (list code
	    (clir-parse-type-annot sm)
	    (clir-parse-unary-exp sm)))
     (else
      (clir-parse-primary-exp sm)))))

(define (clir-parse-primary-exp sm)
  (clir-parse-atomic-exp sm))

(define (clir-parse-atomic-exp sm)
  (let* ((code (case *clir-lex-token*
		 ((sym) 'REG)
		 (else *clir-lex-token*)))
	 (exp (case code
		((lpa)
		 (clir-lex sm)
		 (prog1
		     (clir-parse-exp sm)
		   (clir-lex-expect sm 'rpa)))
		((STATIC FRAME LABEL INTCONST REG)
		 (prog1 `(,code ? ,*clir-lex-value*) (clir-lex sm)))
		;; $B!z$^$@(B:
		;; $B!z(BFLOATCONST
		;; $B!z(BSUBREG
		(else
		 (error "Syntax error"))
		))
	 (type (clir-parse-type-annot sm)))
    (set-car! (cdr exp) type)
    exp))

(define (clir-parse-type-annot sm)
  (cond
   ((eq? *clir-lex-token* 'colon)
    (clir-lex sm)
    (unless (eq? *clir-lex-token* 'sym)
      (error "Syntax error: %s is illegal L-type" *clir-lex-value*))
    ;; $B!z@5Ev$J(B L-type $B$G$"$k$3$H$r%A%'%C%/!#(B
    (prog1
	(string->symbol *clir-lex-value*)
      (clir-lex sm)))
   (else 'I32)))


;;; SLIR -> CLIR

(define (slir->clir exp)
  (slir->clir-exp exp))

(define (clir-paren exp cargs)
  (define (paren x) (format 0 "(%s)" x))
  ;; cargs $B$O(B clir $B$KJQ49$5$l$?(B exp $B$N(B args $B$N%j%9%H!#(B
  ;; $B$3$l$i$KI,MW$J3g8L$r$D$1$k!#(B
  (let* ((code (exp-code exp))
	 (args (exp-args exp))
	 (prec (clir-code-prec code))
	 (codes (map exp-code args))
	 (precs (map clir-code-prec codes)))
    (cond
     ((= 1 prec)
      (if (< prec (car precs))
	  (map paren cargs)
	cargs))
     ((<= 2 prec)
      (let ((left (if (< prec (car precs))
		      (paren (car cargs))
		    (car cargs)))
	    (right (if (<= prec (cadr precs))
		       (paren (cadr cargs))
		     (cadr cargs))))
	(list left right))))))

(define (slir->clir-exp exp)
  (if (atomic-typed-exp? exp)
      (slir->clir-atomic-exp exp)
    (let* ((cargs (clir-paren exp (map-exp-args slir->clir-exp exp)))
	   (prec (clir-code-prec (exp-code exp))))
      (if (= 1 prec)
	  ;; unary
	  (format 0 "%s %s"
		  (slir->clir-exp-code exp)
		  (car cargs))
	;; binary
	(format 0 "%s %s %s"
		(car cargs)
		(slir->clir-exp-code exp)
		(cadr cargs))))))

(define (slir->clir-exp-code exp)
  (format 0 "%s%s"
	  (clir-code-cop (exp-code exp))
	  (slir->clir-exp-type-and-plist exp)))

(define (slir->clir-exp-type-and-plist exp)
  (let ((type (exp-type exp)))
    (format 0 "%s%s"
	    (if (eq? type 'I32) "" (format 0 ":%s" type))
	    (slir->clir-exp-plist exp))))

(define (slir->clir-exp-plist exp)
  (let ((plist (exp-plist exp)))
    (if (null? plist)
	""
      (apply string-append
	     (map (lambda (kv) (format 0 ",%s=%S" (car kv) (cadr kv)))
		  (plist->alist plist))))))
    
(define (slir->clir-atomic-exp exp)
  (let* ((arg (exp-atomic-typed-arg exp))
	 (type (exp-type exp))
	 (clir (case (exp-code exp)
		 ((INTCONST FLOATCONST)
		  (format 0 "%S" arg))
		 ((STATIC)
		  (format 0 "%S" arg))
		 ((FRAME)
		  (format 0 "'%s'" arg))
		 ((LABEL)
		  (format 0 "`%s`" arg))
		 ((REG)
		  (format 0 "%s" arg))
		 ((SUBREG)
		  ;; $B!z$^$@!#(B
		  )
		 (else
		  (error "Syntax error: unexpected: %S" exp)))))
    ;;
    (format 0 "%s%s" clir (slir->clir-exp-type-and-plist exp))))
