;;; $B%i%$%V%i%j!#(B

(define (funcall f . args) (apply f args))

(define (atom? x) (not (pair? x)))

(define (eval-with-env form env)
  (eval `(let ,(map (lambda (e) `(,(car e) ',(cadr e))) env) ,form)))

(define (error fmt . args)
  (runtime-exception
   (format 0 "(error):\n%s\n" (apply format 0 fmt args))))

(define (assert cond)
  (or cond (error "*** ASSERTION FAILED ***\n")))
  
(define (prog1 x . rest) x)

(define (find pred x)
  ;; $B%j%9%H(B x $B$NMWAG$G(B pred $B$rK~$?$9:G=i$N$b$N$r!"$J$1$l$P(B #f $B$rJV$9!#(B
  (cond
   ((null? x) #f)
   ((pred (car x)) (car x))
   (else
    (find pred (cdr x)))))

(define (nub x)
  ;; $B%j%9%H(B x $B$NMWAG$r0l0UE*$K$7$?%j%9%H$rJV$9!#%F%9%H$O(B eqv?
  (if (null? x)
      x
    (let ((y (nub (cdr x))))
      (if (memv (car x) y)
	  y
	(cons (car x) y)))))

(define (butlast x)
  ;; (a b c) -> (a b)
  (cond
   ((null? x) x)
   ((null? (cdr x)) ())
   (else (cons (car x) (butlast (cdr x))))))

(define (copy-sexp sexp)
  (if (pair? sexp)
      (cons (copy-sexp (car sexp))
	    (copy-sexp (cdr sexp)))
    sexp))
    
(define (all-symbols x)
  ;; S-$B<0$K4^$^$l$k%7%s%\%k$N%j%9%H$rJV$9!#(B
  ;; $B0l0U2=$O$7$F$$$J$$!#(B
  (cond
   ((null? x) x)
   ((symbol? x) (list x))
   ((not (pair? x)) ())
   (else
    (append
     (all-symbols (car x))
     (all-symbols (cdr x))))))

(define (zip-with f x y)
  (if (or (null? x) (null? y))
      ()
    (cons (f (car x) (car y))
	  (zip-with f (cdr x) (cdr y)))))

(define (filter f x)
  (cond
   ((null? x) ())
   ((f (car x)) (cons (car x) (filter f (cdr x))))
   (else (filter f (cdr x)))))

(define (count f x)
  (cond
   ((null? x) 0)
   ((f (car x)) (1+ (count f (cdr x))))
   (else (count f (cdr x)))))

;; (define (subst-sexp sexp alist)
;;   (cond
;;    ((symbol? sexp)
;;     (let ((a (assq sexp alist)))
;;       (if a (cadr a) sexp)))
;;    ((pair? sexp)
;;     (cons (subst-sexp (car sexp) alist)
;;           (subst-sexp (cdr sexp) alist)))
;;    (else
;;     sexp)))


(define (replace-sublist x from to)
  (define (match y)
    (let ((y1 y))
      (call/cc
       (lambda (return)
	 (dolist (f from y1)
	   (unless (eqv? f (car y1))
	     (return #f))
	   (set! y1 (cdr y1)))))))
  (if (null? x)
      ()
    (let ((m (match x)))
      (if m
	  (append to (replace-sublist m from to))
	(cons (car x) (replace-sublist (cdr x) from to))))))

(define (replace-substring x from to)
  (list->string
   (replace-sublist
    (string->list x)
    (string->list from)
    (string->list to))))

;;; $BJ8;z$N<oJL!#(B

(define (char-alphabet? c)
  (if (eof-object? c)
      #f
    (or (and (char<=? #\a c) (char<=? c #\z))
	(and (char<=? #\A c) (char<=? c #\Z)))))
  
(define (char-digit? c)
  (if (eof-object? c)
      #f
    (and (char<=? #\0 c) (char<=? c #\9))))
      

;;; $B%j%9%H$KB0@-$r5-O?$9$k$?$a$NJd=u4X?t!#(B
;;; $BB0@-$,$J$$>l9g$G$b(B plist $B$O6u%j%9%H$G$J$$$H2>Dj!#(B

(define (put-prop plist key val)
  ;; $B?7$?$JB0@-$O(B plist $B$N:G8e$KDI2C$5$l$k!#(B
  (let ((place (memq key plist)))
    (if place
	(set-car! (cdr place) val)
      (nconc plist (list key val)))
    plist))

(define (get-prop plist key)
  ;; $B$J$1$l$P%(%i!<!#(B
  (let ((kv (memq key plist)))
    (unless kv (error "%S is not in %S" key plist))
    (cadr kv)))

(define (has-prop? plist key)
  ;; $BB0@-$,$"$k$+$I$&$+%F%9%H!#(B
  (memq key plist))

(define (rem-prop plist key)
  ;; $BB0@-$r:o=|!#(B
  (dolist* (r plist)
    (if (eq? (cadr r) key)
	(set-cdr! r (cdddr r))))
  plist)

(define (plist->alist plist)
  ;; (k1 v1 k2 v2 ...) -> ((k1 v1) (k2 v2) ...)
  (let ((alist ()))
    (do ((kv plist (cddr kv)))
	((null? kv) (nreverse alist))
      (set! alist (cons (list (car kv) (cadr kv)) alist)))))

(define (alist->plist alist)
  ;; ((k1 v1) (k2 v2) ...) -> (k1 v1 k2 v2 ...)
  (apply append alist))
	 
      

;;; $B%U%!%$%kL>4X78!#(B

(define (file-name-split name ch)
  ;; $BJd=u4X?t!#(B
  (let* ((len (string-length name))
	 (idx (do ((i (1- len) (1- i)))
		   ((or (< i 0) (char=? (string-ref name i) ch)) i))))
    (if (< idx 0)
	(list name #f)
      (list (substring name 0 idx)
	    (substring name (1+ idx) len)))))

(define (file-name-suffix name)
  ;; "foo.c" -> ".c"
  ;; "bar"   -> "bar"
  (let ((suf (cadr (file-name-split name #\.))))
    (and suf
	(string-append "." suf))))

(define (file-name-without-suffix name)
  ;; "foo.c" -> "foo"
  ;; "bar"   -> "bar"
  (car (file-name-split name #\.)))

(define (file-name-change-suffix name suf)
  (string-append (file-name-without-suffix name) suf))


;;; IO

(define (read-sexps-from-file file)
  (call-with-input-file file
    (lambda (is)
      (let ((sexp ())
	    (sexps (make-tconc)))
	(while (not (eof-object? (set! sexp (read is))))
	  (tconc sexps sexp))
	(tconc->list sexps)))))


    
;;; format $B4X?t!#(B

(define (format sm fmt . args)
  ;;  %[<field>{.<field>}][<padc>]<fmtc>
  ;;  <field>  ::= [-]{0|1|...|9} | #  (printf $B$HF1MM!#(B# $B$O0z?t$+$i<h$k(B)
  ;;  <fmtc>   ::=  $B#1J8;z(B ($B%U%)!<%^%C%H$N;XDj(B)
  ;;  <padc>   ::= '$B#1J8;z(B ($B%Q%G%#%s%0J8;z!"%G%U%)%k%H$O6uGr(B)
  ;;
  ;;  sm = 0 $B$J$iJ8;zNs$H$7$F!"(B1 $B$J$iI8=`=PNO!"(B
  ;;  $B$=$l0J30$J$i%9%H%j!<%`(B sm $B$K=PNO!#(B
  ;;
  (let ((os (case sm
	      ((0) (open-output-string))
	      ((1) (current-output-port))
	      (else
	       (or (output-port? sm)
		   (error "format: illegal 1st argument"))
	       sm)))
	(is (open-input-string fmt)))
    (while (not (eof-object? (peek-char is)))
      (case (peek-char is)
	((#\%)
	 (read-char is)
	 (set! args (format-apply os (format-parse is) args)))
	(else
	 (write-char (read-char is) os))))
    (or (null? args)
	(error "format: too many arguments"))
    (close-input-port is)
    (if (eqv? sm 0)
	(prog1
	 (get-output-string os)
	 (close-output-port os)))))

(define (format-parse is)
  (let ((field (make-tconc))
	(padc #\space)
	(fmtc ()))
    (tconc field (format-parse-field is))
    (while (eqv? (peek-char is) #\.)
      (read-char is)
      (tconc field (format-parse-field is)))
    (if (eqv? (peek-char is) #\')
	(begin
	  (read-char is)
	  (set! padc (read-char is))))
    (set! fmtc (read-char is))
    (list (tconc->list field) padc fmtc)))

(define (format-parse-field is)
  (define (digits? c)
    (and (char? c) (char<=? #\0 c) (char<=? c #\9)))
  (if (eqv? (peek-char is) #\#)
      (read-char is)
    (let ((sign 1)
	  (num 0))
      (if (eqv? (peek-char is) #\-)
	  (begin
	    (read-char is)
	    (set! sign -1)))
      (while (digits? (peek-char is))
	(set! num (+ (* num 10)
		     (- (char->integer (read-char is))
			(char->integer #\0)))))
      (* sign num))))

(define (format-apply os fm args)
  (define (anyarg? x) #t)
  (let ((field (car fm))
	(padc (cadr fm))
	(fmtc (caddr fm)))
    (dolist* (f field)
      (if (eqv? (car f) #\#)
	  (begin
	    (format-check-args args integer?)
	    (set-car! f (car args))
	    (set! args (cdr args)))))
    (case fmtc
      ((#\%) ; %%  1 $B$D$N(B % $B$r=PNO!#(B
       (write-char #\% os)
       args)
      ((#\s) ; %[<field>]s $B:G>.I}(B <field> $B$G(B S-$B<0$r(B display$B!#(B
       (format-check-args args anyarg?)
       (format-fill os (car field) padc (display-to-string (car args)))
       (cdr args))
      ((#\S) ; %[<field>]s $B:G>.I}(B <field> $B$G(B S-$B<0$r(B write$B!#(B
       (format-check-args args anyarg?)
       (format-fill os (car field) padc (write-to-string (car args)))
       (cdr args))
      ((#\c) ; %[<field>]c $B:G>.I}(B <field> $B$GJ8;z$r=PNO!#(B
       (format-check-args args char?)
       (format-fill os (car field) padc (display-to-string (car args)))
       (cdr args))
      ((#\d) ; %[<field>]d $B:G>.I}(B <field> $B$G@0?t$r(B 10 $B?J=PNO(B($BIi$J$i(B - $BId9g(B)$B!#(B
       (format-check-args args integer?)
       (format-fill os (car field) padc (number->string (car args)))
       (cdr args))
      ((#\x #\X #\o #\b) ; %[<field>]? $BI}(B <field> $B$G@0?t$r(B (16,8,2) $B?J=PNO!#(B
       (format-check-args args integer?)
       (format-fill os (car field) padc
		    (format-xob (car args) fmtc (abs (car field))))
       (cdr args))
      ;; $B!z(B float ...
      (else
       (if (eof-object? fmtc)
	   ;; (format sm ".... %") $B$N>l9g!"(Bflush-port $B$9$k!#(B
	   (begin
	     (flush-port os)
	     args)
	 (error "format: %%%c is illegal format" fmtc))))))

(define (format-check-args args . preds)
  (if (< (length args) (length preds))
      (error "format: too few arguments"))
  (dolist (p preds)
    (if (not (p (car args)))
	(error "format: illegal argument"))
    (set! args (cdr args))))

(define (format-fill os width padc str)
  (display
   (let ((len (string-length str)))
     (cond
      ((<= (abs width) len)
       str)
      ((<= 0 width)
       (string-append (make-string (- width len) padc) str))
      ((< width 0)
       (string-append str (make-string (- (- width) len) padc)))))
   os))

(define (format-xob num fmtc ndig)
  (let* ((w (case fmtc ((#\x #\X) 4) ((#\o) 3) ((#\b) 1)))
	 (mask (- (logshl 1 w) 1))
	 (dig (if (char=? fmtc #\X) "0123456789ABCDEF" "0123456789abcdef"))
	 (diglist ()))
    (if (and (< num 0) (zero? ndig))
	(error "format: default field width of xob for negative value"))
    (if (zero? ndig)
	(set! ndig 80))
    (call/cc
     (lambda (break)
       (dotimes (i ndig)
	 (if (zero? num) (break ()))
	 (set! diglist (cons (string-ref dig (logand num mask)) diglist))
	 (set! num (logshr num w)))))
    (list->string diglist)))

(define (call-with-output-string f)
  (let ((os (open-output-string)))
    (f os)
    (prog1
     (get-output-string os)
     (close-output-port os))))

(define (write-to-string x)
  (call-with-output-string
   (lambda (sm) (write x sm))))

(define (display-to-string x)
  (call-with-output-string
   (lambda (sm) (display x sm))))


;;; $BHFMQ%D%j!<I=<(4X?t!#(B

(define (print-tree sm node nodeargs printnode . ihook-opt)
  ;; $B%9%H%j!<%`(B sm $B$K%D%j!<$rI=<(!#(Bsm=1 $B$J$i(B (current-output-port)$B!#(B
  ;; node $B$O%D%j!<$rI=$9G$0U$N(B Lisp $B%*%V%8%'%/%H!#(B
  ;; nodeargs $B$O(B (nodeargs node) $B$H8F$P$l!"(Bnode $B$N;R$N%j%9%H$rJV$94X?t!#(B
  ;; printnode $B$O(B (printnode sm node) $B$H8F$P$l!"%9%H%j!<%`(B sm $B$K(B node
  ;; $B$N>pJs$r=q$/!#:G=i$N(B 1 $B9T$H(B 2 $B9TL\0J9_(B($B$b$7$"$l$P(B)$B$OI=<($N$5$l$+$?$,(B
  ;; $B0[$J$k!#Nc(B: (example-of-print-tree)$B!#(B
  ;;
  (define (split-string str)
    (define (split-list elm x)
      (cond
       ((null? x) (list ()))
       ((eqv? (car x) elm) (cons () (split-list elm (cdr x))))
       (else
	(let ((r (split-list elm (cdr x))))
	  (cons (cons (car x) (car r)) (cdr r))))))
    ;;
    (map list->string (split-list #\newline (string->list str))))
  ;;
  (define (print-tree-aux sm node nodeargs printnode indent child ihook)
    (let ((args (nodeargs node))
	  (info (split-string
		 (call-with-output-string
		  (lambda (sm) (printnode sm node))))))
      (ihook sm)
      (if child
	  (format sm "%s+--" (substring indent 0 (- (string-length indent) 3)))
	(format sm "%s" indent))
      (format sm "%s\n" (car info))
      (dolist (i (cdr info))
	(ihook sm)
	(format sm "%s%s%s\n" indent (if (null? args) "  " "| ") i))
      (dolist* (as args)
	(print-tree-aux
	 sm (car as) nodeargs printnode
	 (string-append indent (if (null? (cdr as)) "   " "|  "))
	 #t
	 ihook))))
  ;;
  (print-tree-aux sm node nodeargs printnode "" #f
		  (if (null? ihook-opt) identity (car ihook-opt))))
						      
(define (print-tree-with-linenum sm node nodeargs printnode)
  ;; $B;2>HMQ$NHV9f$D$-(B print-tree
  (let ((line 0))
    (define (ihook sm) (format sm "%3d  " (set! line (1+ line))))
    (print-tree sm node nodeargs printnode ihook)))

(define (example-of-print-tree)
  ;; print-tree $B$N;HMQNc!#(B
  ;; $B0J2<$N$h$&$KI=<($5$l$k!#(B
  ;;
  ;; Node a
  ;; | x=1
  ;; | y=0
  ;; +--Node b
  ;; |  | x=1
  ;; |  +--Node c
  ;; |  |  | y=0
  ;; |  |  +--Node d
  ;; |  +--Node e
  ;; |       x=1
  ;; |       y=2
  ;; |       z=1
  ;; +--Node f
  ;;      z=2
  ;;
  (print-tree 1  ; (current-output-port)
	      '(a ("x=1" "y=0")
		  (b ("x=1")
		     (c ("y=0")
			(d ()))
		     (e ("x=1" "y=2" "z=1")))
		  (f ("z=2")))
	      cddr
	      (lambda (sm node)
		(format sm "Node %s" (car node))
		(dolist (i (cadr node))
		  (format sm "\n%s" i)))))
